<p>Rejecting requests with significant content length is a good practice to control the network traffic intensity and thus resource consumption in
order to prevent DoS attacks.</p>
<h2>Ask Yourself Whether</h2>
<ul>
  <li> size limits are not defined for the different resources of the web application. </li>
  <li> the web application is not protected by <a href="https://en.wikipedia.org/wiki/Rate_limiting">rate limiting</a> features. </li>
  <li> the web application infrastructure has limited resources. </li>
</ul>
<p>There is a risk if you answered yes to any of those questions.</p>
<h2>Recommended Secure Coding Practices</h2>
<ul>
  <li> For most of the features of an application, it is recommended to limit the size of requests to:
    <ul>
      <li> lower or equal to 8mb for file uploads. </li>
      <li> lower or equal to 2mb for other requests. </li>
    </ul>  </li>
</ul>
<p>It is recommended to customize the rule with the limit values that correspond to the web application.</p>
<h2>Sensitive Code Example</h2>
<pre>
using Microsoft.AspNetCore.Mvc;

public class MyController : Controller
{
    [HttpPost]
    [DisableRequestSizeLimit] // Sensitive: No size  limit
    [RequestSizeLimit(10485760)] // Sensitive: 10485760 B = 10240 KB = 10 MB is more than the recommended limit of 8MB
    public IActionResult PostRequest(Model model)
    {
    // ...
    }

    [HttpPost]
    [RequestFormLimits(MultipartBodyLengthLimit = 10485760)] // Sensitive: 10485760 B = 10240 KB = 10 MB is more than the recommended limit of 8MB
    public IActionResult MultipartFormRequest(Model model)
    {
    // ...
    }
}
</pre>
<p>In Web.config:</p>
<pre>
&lt;configuration&gt;
    &lt;system.web&gt;
        &lt;httpRuntime maxRequestLength="81920" executionTimeout="3600" /&gt;
        &lt;!-- Sensitive: maxRequestLength is expressed in KB, so 81920 KB = 80 MB  --&gt;
    &lt;/system.web&gt;
    &lt;system.webServer&gt;
        &lt;security&gt;
            &lt;requestFiltering&gt;
                &lt;requestLimits maxAllowedContentLength="83886080" /&gt;
                &lt;!-- Sensitive: maxAllowedContentLength is expressed in bytes, so 83886080 B = 81920 KB = 80 MB  --&gt;
            &lt;/requestFiltering&gt;
        &lt;/security&gt;
    &lt;/system.webServer&gt;
&lt;/configuration&gt;
</pre>
<h2>Compliant Solution</h2>
<pre>
using Microsoft.AspNetCore.Mvc;

public class MyController : Controller
{
    [HttpPost]
    [RequestSizeLimit(8388608)] // Compliant: 8388608 B = 8192 KB = 8 MB
    public IActionResult PostRequest(Model model)
    {
    // ...
    }

    [HttpPost]
    [RequestFormLimits(MultipartBodyLengthLimit = 8388608)] // Compliant: 8388608 B = 8192 KB = 8 MB
    public IActionResult MultipartFormRequest(Model model)
    {
    // ...
    }
}
</pre>
<p>In Web.config:</p>
<pre>
&lt;configuration&gt;
    &lt;system.web&gt;
        &lt;httpRuntime maxRequestLength="8192" executionTimeout="3600" /&gt;
        &lt;!-- Compliant: maxRequestLength is expressed in KB, so 8192 KB = 8 MB  --&gt;
    &lt;/system.web&gt;
    &lt;system.webServer&gt;
        &lt;security&gt;
            &lt;requestFiltering&gt;
                &lt;requestLimits maxAllowedContentLength="8388608" /&gt;
                &lt;!-- Compliant: maxAllowedContentLength is expressed in bytes, so 8388608 B = 8192 KB = 8 MB  --&gt;
            &lt;/requestFiltering&gt;
        &lt;/security&gt;
    &lt;/system.webServer&gt;
&lt;/configuration&gt;
</pre>
<h2>See</h2>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A05_2021-Security_Misconfiguration/">Top 10 2021 Category A5 - Security Misconfiguration</a> </li>
  <li> <a href="https://cheatsheetseries.owasp.org/cheatsheets/Denial_of_Service_Cheat_Sheet.html">Owasp Cheat Sheet</a> - Owasp Denial of Service
  Cheat Sheet </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration">Top 10 2017 Category A6 - Security
  Misconfiguration</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/770">CWE-770 - Allocation of Resources Without Limits or Throttling</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/400">CWE-400 - Uncontrolled Resource Consumption</a> </li>
  <li> <a href="https://docs.microsoft.com/en-us/aspnet/core/host-and-deploy/iis/web-config">Web.config</a> - XML-formatted config file for IIS
  applications </li>
</ul>

