<h2>Why is this an issue?</h2>
<p>It is equivalent to use the equality <code>==</code> operator and the <code>equals</code> method to compare two objects if the <code>equals</code>
method inherited from <code>Object</code> has not been overridden. In this case both checks compare the object references.</p>
<p>But as soon as <code>equals</code> is overridden, two objects not having the same reference but having the same value can be equal. This rule spots
suspicious uses of <code>==</code> and <code>!=</code> operators on objects whose <code>equals</code> methods are overridden.</p>
<h3>Noncompliant code example</h3>

```java
String firstName = getFirstName(); // String overrides equals
String lastName = getLastName();

if (firstName == lastName) { ... }; // Non-compliant; false even if the strings have the same value
```
<h3>Compliant solution</h3>

```java
String firstName = getFirstName();
String lastName = getLastName();

if (firstName != null && firstName.equals(lastName)) { ... };
```
<h3>Exceptions</h3>
<p>Comparing two instances of the <code>Class</code> object will not raise an issue:</p>

```java
Class c;
if(c == Integer.class) { // No issue raised
}
```
<p>Comparing <code>Enum</code> will not raise an issue:</p>

```java
public enum Fruit {
   APPLE, BANANA, GRAPE
}
public boolean isFruitGrape(Fruit candidateFruit) {
  return candidateFruit == Fruit.GRAPE; // it's recommended to activate S4551 to enforce comparison of Enums using ==
}
```
<p>Comparing with <code>final</code> reference will not raise an issue:</p>

```java
private static final Type DEFAULT = new Type();

void foo(Type other) {
  if (other == DEFAULT) { // Compliant
  //...
  }
}
```
<p>Comparing with <code>this</code> will not raise an issue:</p>

```java
  public boolean equals(Object other) {
    if (this == other) {  // Compliant
      return false;
    }
 }
```
<p>Comparing with <code>java.lang.String</code> and boxed types <code>java.lang.Integer</code>, …​ will not raise an issue.</p>
<h2>Resources</h2>
<ul>
  <li> <a href='checks/checks/?languages=JAVA&tools=SONAR_LINT&rule=java:S4973'>java:S4973</a> - Strings and Boxed types should be compared using "equals()" </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/595">CWE-595 - Comparison of Object References Instead of Object Contents</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/597">CWE-597 - Use of Wrong Operator in String Comparison</a> </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/UjdGBQ">CERT, EXP03-J.</a> - Do not use the equality operators when comparing values of boxed
  primitives </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/yDdGBQ">CERT, EXP50-J.</a> - Do not confuse abstract object equality with reference equality
  </li>
</ul>