<h2>Why is this an issue?</h2>
<p>Servlets are components in Java web development, responsible for processing HTTP requests and generating responses. In this context, exceptions are
used to handle and manage unexpected errors or exceptional conditions that may occur during the execution of a servlet.</p>
<p>Catching exceptions within the servlet allows us to convert them into meaningful, user-friendly messages. Otherwise, failing to catch exceptions
will propagate them to the servlet container, where the default error-handling mechanism may impact the overall security and stability of the
server.</p>
<p>Possible security problems are:</p>
<ol>
  <li> <strong>Vulnerability to denial-of-service attacks:</strong> Not caught exceptions can leave the servlet container in an unstable state, which
  can exhaust the available resources and make the system unavailable in the worst cases. </li>
  <li> <strong>Exposure of sensitive information:</strong> Exceptions handled by the servlet container, by default, expose detailed error messages or
  debugging information to the user, which may contain sensitive data such as stack traces, database connection, or system configuration. </li>
</ol>
<p>Unfortunately, servlet method signatures do not force developers to handle <code>IOException</code> and <code>ServletException</code>:</p>

```java
public void doGet(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
}
```
<p>To prevent this risk, this rule enforces all exceptions to be caught within the "do*" methods of servlet classes.</p>
<h2>How to fix it</h2>
<p>Surround all method calls that may throw an exception with a <code>try/catch</code> block.</p>
<h3>Code examples</h3>
<p>In the following example, the <code>getByName</code> method may throw an <code>UnknownHostException</code>.</p>
<h4>Noncompliant code example</h4>

```java
public void doGet(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
  InetAddress addr = InetAddress.getByName(request.getRemoteAddr()); // Noncompliant
  //...
}
```
<h4>Compliant solution</h4>

```java
public void doGet(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
  try {
    InetAddress addr = InetAddress.getByName(request.getRemoteAddr());
    //...
  }
  catch (UnknownHostException ex) {  // Compliant
    //...
  }
}
```
<h2>Resources</h2>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/600">CWE-600 - Uncaught Exception in Servlet</a> </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/-zZGBQ">CERT, ERR01-J.</a> - Do not allow exceptions to expose sensitive information </li>
</ul>