<p>This vulnerability increases the likelihood that attackers are able to compute the cleartext of password hashes.</p>
<h2>Why is this an issue?</h2>
<p>During the process of password hashing, an additional component, known as a "salt," is often integrated to bolster the overall security. This salt,
acting as a defensive measure, primarily wards off certain types of attacks that leverage pre-computed tables to crack passwords.</p>
<p>However, potential risks emerge when the salt is deemed insecure. This can occur when the salt is consistently the same across all users or when it
is too short or predictable. In scenarios where users share the same password and salt, their password hashes will inevitably mirror each other.
Similarly, a short salt heightens the probability of multiple users unintentionally having identical salts, which can potentially lead to identical
password hashes. These identical hashes streamline the process for potential attackers to recover clear-text passwords. Thus, the emphasis on
implementing secure, unique, and sufficiently lengthy salts in password-hashing functions is vital.</p>
<h3>What is the potential impact?</h3>
<p>Despite best efforts, even well-guarded systems might have vulnerabilities that could allow an attacker to gain access to the hashed passwords.
This could be due to software vulnerabilities, insider threats, or even successful phishing attempts that give attackers the access they need.</p>
<p>Once the attacker has these hashes, they will likely attempt to crack them using a couple of methods. One is brute force, which entails trying
every possible combination until the correct password is found. While this can be time-consuming, having the same salt for all users or a short salt
can make the task significantly easier and faster.</p>
<p>If multiple users have the same password and the same salt, their password hashes would be identical. This means that if an attacker successfully
cracks one hash, they have effectively cracked all identical ones, granting them access to multiple accounts at once.</p>
<p>A short salt, while less critical than a shared one, still increases the odds of different users having the same salt. This might create clusters
of password hashes with identical salt that can then be attacked as explained before.</p>
<p>With short salts, the probability of a collision between two users' passwords and salts couple might be low depending on the salt size. The shorter
the salt, the higher the collision probability. In any case, using longer, cryptographically secure salt should be preferred.</p>
<h3>Exceptions</h3>
<p>To securely store password hashes, it is a recommended to rely on key derivation functions that are computationally intensive. Examples of such
functions are:</p>
<ul>
  <li> Argon2 </li>
  <li> PBKDF2 </li>
  <li> Scrypt </li>
  <li> Bcrypt </li>
</ul>
<p>When they are used for password storage, using a secure, random salt is required.</p>
<p>However, those functions can also be used for other purposes such as master key derivation or password-based pre-shared key generation. In those
cases, the implemented cryptographic protocol might require using a fixed salt to derive keys in a deterministic way. In such cases, using a fixed
salt is safe and accepted.</p>
<h2>How to fix it in Java SE</h2>
<h3>Code examples</h3>
<p>The following code contains examples of hard-coded salts.</p>
<h4>Noncompliant code example</h4>

```java
import javax.crypto.spec.PBEParameterSpec;

public void hash() {
    byte[] salt = "salty".getBytes();
    PBEParameterSpec cipherSpec = new PBEParameterSpec(salt, 10000); // Noncompliant
}
```
<h4>Compliant solution</h4>

```java
import java.security.SecureRandom;
import javax.crypto.spec.PBEParameterSpec;

public void hash() {
    SecureRandom random = new SecureRandom();
    byte[] salt = new byte[32];
    random.nextBytes(salt);

    PBEParameterSpec cipherSpec = new PBEParameterSpec(salt, 10000);
}
```
<h3>How does this work?</h3>
<p>This code ensures that each user’s password has a unique salt value associated with it. It generates a salt randomly and with a length that
provides the required security level. It uses a salt length of at least 32 bytes (256 bits), as recommended by industry standards.</p>
<p>Here, the compliant code example ensures the salt is random and has a sufficient length by calling the <code>nextBytes</code> method from the
<code>SecureRandom</code> class with a salt buffer of 16 bytes. This class implements a cryptographically secure pseudo-random number generator.</p>
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">Top 10 2021 Category A2 - Cryptographic Failures</a> </li>
  <li> OWASP - <a href="https://www.owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/759">CWE-759 - Use of a One-Way Hash without a Salt</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/760">CWE-760 - Use of a One-Way Hash with a Predictable Salt</a> </li>
  <li> STIG Viewer - <a href="https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222542">Application Security and
  Development: V-222542</a> - The application must only store cryptographic representations of passwords. </li>
</ul>