<h2>Why is this an issue?</h2>
<p>Using boxed values in a ternary operator does not simply return one operand or the other based on the condition. Instead, the values are unboxed
and coerced to a common type, which can result in a loss of precision when converting one operand from <code>int</code> to <code>float</code> or from
<code>long</code> to <code>double</code>.</p>
<p>While this behavior is expected for arithmetic operations, it may be unexpected for the ternary operator. To avoid confusion or unexpected
behavior, cast to a compatible type explicitly.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>Cast one of both operands to a common supertype (e.g., <code>Number</code>) to prevent auto-unboxing and, thus, type coercion.</p>

```java
Integer i = 123456789;
Float f = 1.0f;
Number n1 = condition ? i : f;  // Noncompliant, unexpected precision loss, n1 = 1.23456792E8
```
<h4>Compliant solution</h4>

```java
Integer i = 123456789;
Float f = 1.0f;
Number n1 = condition ? (Number) i : f; // Compliant, cast to Number prevents unboxing
Number n2 = condition ? i : (Number) f; // Compliant, cast to Number prevents unboxing
```
<h4>Noncompliant code example</h4>
<p>If type coercion was your intention, clarify this by casting the operand that would be coerced to the corresponding type explicitly.</p>

```java
Integer i = 123456789;
Float f = 1.0f;
Number n1 = condition ? i : f;  // Noncompliant, unexpected precision loss, n1 = 1.23456792E8
```
<h4>Compliant solution</h4>

```java
Integer i = 123456789;
Float f = 1.0f;
Number n = condition ? (float) i : f; // Compliant, intentional type coercion with precision loss
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://docs.oracle.com/javase/tutorial/java/nutsandbolts/op2.html">The Java Tutorials: Equality, Relational, and Conditional
  Operators</a> </li>
  <li> <a href="https://docs.oracle.com/javase/tutorial/java/data/autoboxing.html">The Java Tutorials: Autoboxing and Unboxing</a> </li>
  <li> <a href="https://docs.oracle.com/javase/specs/jls/se7/html/jls-5.html">The Java® Language Specification Java SE 7 Edition: Chapter 5.
  Conversions and Promotions</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://www.geeksforgeeks.org/coercion-in-java/">GeeksforGeeks: Coercion in Java</a> </li>
</ul>