<h2>Why is this an issue?</h2>
<p>The JDK provides a set of built-in methods to copy the contents of an array into another array. Using a loop to perform the same operation is less
clear, more verbose and should be avoided.</p>
<h3>Exceptions</h3>
<p>The rule detects only the most idiomatic patterns, it will not consider loops with non-trivial control flow. For example, loops that copy array
elements conditionally are ignored.</p>
<h2>How to fix it</h2>
<p>You can use:</p>
<ul>
  <li> <code>Arrays.copyOf</code> to copy an entire array into another array </li>
  <li> <code>System.arraycopy</code> to copy only a subset of an array into another array </li>
  <li> <code>Arrays.asList</code> to create a new list with the contents of the array </li>
  <li> <code>Collections.addAll</code> to add the elements of a collection into another collection </li>
</ul>
<p>Note that <code>Arrays.asList</code> returns a fixed-size <code>List</code>, so further steps are required if a non-fixed-size <code>List</code> is
needed.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
public void copyArray(String[] source){
  String[] array = new String[source.length];
  for (int i = 0; i < source.length; i++) {
    array[i] = source[i]; // Noncompliant
  }
}

public void copyList(List<String> source) {
  List<String> list = new ArrayList<>();
  for (String s : source) {
    list.add(s); // Noncompliant
  }
}
```
<h4>Compliant solution</h4>

```java
public void copyArray(String[] source){
  String[] array = Arrays.copyOf(source, source.length);
}

public void copyList(List<String> source) {
  List<String> list = new ArrayList<>();
  Collections.addAll(list, source);
}
```

```java
public void makeCopiesConditional(int[] source) {
  int[] dest = new int[source.length];
  for (int i = 0; i < source.length; i++) {
    if (source[i] > 10) {
      dest[i] = source[i];  // Compliant, since the array elements are conditionally copied to the dest array
    }
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/util/Arrays.html#copyOf-T:A-int-:~:text=1.6-,copyOf,-public%20static%C2%A0byte">docs.oracle</a>
  - Arrays.copyOf documentation </li>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/util/Arrays.html#asList-T…​-:~:text=1.6-,asList,-%40SafeVarargs%0Apublic%20static">docs.oracle</a> - Arrays.asList documentation </li>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/lang/System.html#arraycopy-java.lang.Object-int-java.lang.Object-int-int-:~:text=1.5-,arraycopy,-public%20static%C2%A0void">docs.oracle</a> - System.arraycopy documentation </li>
</ul>