<h2>Why is this an issue?</h2>
<p>A <code>printf-</code>-style format string is a string that contains placeholders, usually represented by special characters such as "%s" or "{}",
depending on the technology in use. These placeholders are replaced by values when the string is printed or logged.</p>
<p>Because <code>printf</code>-style format strings are interpreted at runtime, rather than validated by the compiler, they can contain errors that
result in the wrong strings being created.</p>
<p>This rule checks whether every format string specifier can be correctly matched with one of the additional arguments when calling the following
methods:</p>
<ul>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/lang/String.html#format-java.lang.String-java.lang.Object…​-"><code>java.lang.String#format</code></a> </li>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/util/Formatter.html#format-java.lang.String-java.lang.Object…​-"><code>java.util.Formatter#format</code></a> </li>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/io/PrintStream.html#format-java.lang.String-java.lang.Object…​-"><code>java.io.PrintStream#format</code></a> </li>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/text/MessageFormat.html#format-java.lang.String-java.lang.Object…​-"><code>java.text.MessageFormat#format</code></a> </li>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/io/PrintWriter.html#format-java.lang.String-java.lang.Object…​-"><code>java.io.PrintWriter#format</code></a> </li>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/io/PrintStream.html#printf-java.lang.String-java.lang.Object…​-"><code>java.io.PrintStream#printf</code></a> </li>
  <li> <a
  href="https://docs.oracle.com/javase/8/docs/api/java/io/PrintWriter.html#printf-java.lang.String-java.lang.Object…​-"><code>java.io.PrintWriter#printf</code></a> </li>
  <li> <a
  href="https://docs.oracle.com/en/java/javase/17/docs/api/java.base/java/lang/String.html#formatted(java.lang.Object…​)"><code>java.lang.String#formatted</code></a> (since Java 15) </li>
  <li> logging methods of <a href="https://www.slf4j.org/api/org/slf4j/Logger.html"><code>org.slf4j.Logger</code></a>, <a
  href="https://docs.oracle.com/javase/8/docs/api/java/util/logging/Logger.html"><code>java.util.logging.Logger</code></a>, <a
  href="https://logging.apache.org/log4j/2.x/javadoc/log4j-api/org/apache/logging/log4j/Logger.html"><code>org.apache.logging.log4j.Logger</code></a>.
  </li>
</ul>
<h2>How to fix it</h2>
<p>A <code>printf-</code>-style format string is a string that contains placeholders, which are replaced by values when the string is printed or
logged. Mismatch in the format specifiers and the arguments provided can lead to incorrect strings being created.</p>
<p>To avoid issues, a developer should ensure that the provided arguments match format specifiers.</p>
<p>Note that <a href="https://docs.oracle.com/en/java/javase/21/docs/api/java.base/java/text/MessageFormat.html">MessageFormat</a> is used by most
logging mechanisms, for example <code>java.util.logging.Logger</code>, thus the <em>single quote</em> must be escaped by a <em>double single
quote</em>.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
void logging(org.slf4j.Logger slf4jLog, java.util.logging.Logger logger) {
    String.format("Too many arguments %d and %d", 1, 2, 3); // Noncompliant - the third argument '3' is unused
    String.format("First {0} and then {1}", "foo", "bar");  //Noncompliant - it appears there is confusion with the use of "java.text.MessageFormat" - parameters "foo" and "bar" will be ignored here


    slf4jLog.debug("The number: ", 1); // Noncompliant - String contains no format specifiers.

    logger.log(level, "Can't load library \"{0}\"!", "foo"); // Noncompliant - the single quote ' must be escaped
}
```
<h4>Compliant solution</h4>

```java
void logging(org.slf4j.Logger slf4jLog, java.util.logging.Logger logger) {
    String.format("Too many arguments %d and %d", 1, 2);
    String.format("First %s and then %s", "foo", "bar");

    slf4jLog.debug("The number: {}", 1);

    logger.log(level, "Can''t load library \"{0}\"!", "foo");
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/J9YxBQ">CERT, FIO47-C.</a> - Use valid format strings </li>
  <li> <a href="https://docs.oracle.com/en/java/javase/21/docs/api/java.base/java/text/MessageFormat.html">java.text.MessageFormat</a> </li>
</ul>