<h2>Why is this an issue?</h2>
<p>The <code>Any#equals(other: Any?)</code> method is used to compare two objects to see if they are equal.</p>
<p>The <code>other</code> parameter’s type is <code>Any?</code>, this means that an object of any type, as well as <code>null</code>, can be passed as
an argument to this method.</p>
<p>Any class overriding <code>Any#equals(other: Any?)</code> should respect this contract, accept any object as an argument, and return
<code>false</code> when the argument’s type differs from the expected type. The <code>other</code> parameter’s type can be checked using the
<code>is</code> operator or by comparing the <code>javaClass</code> field:</p>

```kotlin
override fun equals(other: Any?): Boolean {
  // ...
  if (other?.javaClass != this.javaClass) {
    return false
  }
  // ...
}
```
<p>However, it is an issue to assume that the <code>equals</code> method will only be used to compare objects of the same type. Casting the
<code>other</code> parameter without a prior test will throw a <code>ClassCastException</code> instead of returning false.</p>

```kotlin
class MyClass {
  override fun equals(other: Any?): Boolean {
    val that = other as MyClass // may throw a ClassCastException
    // ...
  }
  // ...
}
```
<p>This rule raises an issue when <code>other</code> parameter’s type has not been tested before a cast operation.</p>
<h2>How to fix it</h2>
<p>Ensure the <code>other</code> parameter’s type is checked by comparing <code>other?.javaClass</code> and <code>this.javaClass</code>, or use the
<code>is</code> operator to test `other’s type.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```kotlin
class MyClass {
  override fun equals(other: Any?): Boolean {
    if (this === other) {
        return true
    }
    val that = other as MyClass // Noncompliant, may throw a ClassCastException
    // ...
  }
  // ...
}
```
<h4>Compliant solution</h4>

```kotlin
class MyClass {
  override fun equals(other: Any?): Boolean {
    if (this === other) {
        return true
    }
    if (other?.javaClass != this.javaClass) {
      return false
    }
    val that = other as MyClass // Compliant, other's type is MyClass
    // ...
  }
  // ...
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-any/equals.html">Kotlin Standard Library - Any#equals(other: Any?)</a> </li>
</ul>