A strict mode directive is a `"use strict"` literal at the beginning of a script or function body. It enables strict mode semantics.

When a directive occurs in global scope, strict mode applies to the entire script:

```js
"use strict";

// strict mode

function foo() {
    // strict mode
}
```

When a directive occurs at the beginning of a function body, strict mode applies only to that function, including all contained functions:

```js
function foo() {
    "use strict";
    // strict mode
}

function foo2() {
    // not strict mode
};

(function() {
    "use strict";
    function bar() {
        // strict mode
    }
}());
```

In the **CommonJS** module system, a hidden function wraps each module and limits the scope of a "global" strict mode directive.

In **ECMAScript** modules, which always have strict mode semantics, the directives are unnecessary.

## Rule Details

This rule requires or disallows strict mode directives.

This rule disallows strict mode directives, no matter which option is specified, if ESLint configuration specifies either of the following as [parser options](https://eslint.org/docs/use/configure/language-options#specifying-parser-options):

* `"sourceType": "module"` that is, files are **ECMAScript** modules
* `"impliedStrict": true` property in the `ecmaFeatures` object

This rule disallows strict mode directives, no matter which option is specified, in functions with non-simple parameter lists (for example, parameter lists with default parameter values) because that is a syntax error in **ECMAScript 2016** and later. See the examples of the "function" option.

This rule does not apply to class static blocks, no matter which option is specified, because class static blocks do not have directives. Therefore, a `"use strict"` statement in a class static block is not a directive, and will be reported by the [no-unused-expressions](https://eslint.org/docs/rules/no-unused-expressions) rule.

The `--fix` option on the command line does not insert new `"use strict"` statements, but only removes unneeded statements.

## When Not To Use It

In a codebase that has both strict and non-strict code, either turn this rule off, or [selectively disable it](https://eslint.org/docs/use/configure/rules#disabling-rules) where necessary. For example, functions referencing `arguments.callee` are invalid in strict mode. A [full list of strict mode differences](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Strict_mode/Transitioning_to_strict_mode#Differences_from_non-strict_to_strict) is available on MDN.
