<h2>Why is this an issue?</h2>
<p>When you call <code>Any()</code>, it clearly communicates the code’s intention, which is to check if the collection is empty. Using <code>Count()
== 0</code> for this purpose is less direct and makes the code slightly more complex. However, there are some cases where special attention should be
paid:</p>
<ul>
  <li> if the collection is an <code>EntityFramework</code> or other ORM query, calling <code>Count()</code> will cause executing a potentially
  massive SQL query and could put a large overhead on the application database. Calling <code>Any()</code> will also connect to the database, but will
  generate much more efficient SQL. </li>
  <li> if the collection is part of a LINQ query that contains <code>Select()</code> statements that create objects, a large amount of memory could be
  unnecessarily allocated. Calling <code>Any()</code> will be much more efficient because it will execute fewer iterations of the enumerable. </li>
</ul>
<pre data-diff-id="1" data-diff-type="noncompliant">
private static bool HasContent(IEnumerable&lt;string&gt; strings)
{
  return strings.Count() &gt; 0;  // Noncompliant
}

private static bool HasContent2(IEnumerable&lt;string&gt; strings)
{
  return strings.Count() &gt;= 1;  // Noncompliant
}

private static bool IsEmpty(IEnumerable&lt;string&gt; strings)
{
  return strings.Count() == 0;  // Noncompliant
}
</pre>
<p>Prefer using <code>Any()</code> to test for emptiness over <code>Count()</code>.</p>
<pre data-diff-id="1" data-diff-type="compliant">
private static bool HasContent(IEnumerable&lt;string&gt; strings)
{
  return strings.Any();
}

private static bool HasContent2(IEnumerable&lt;string&gt; strings)
{
  return strings.Any();
}

private static bool IsEmpty(IEnumerable&lt;string&gt; strings)
{
  return !strings.Any();
}
</pre>

