<h2>Why is this an issue?</h2>
<p>Most developers expect property access to be as efficient as field access. However, if a property returns a copy of an array or collection, it will
be much slower than a simple field access, contrary to the caller’s likely expectations. Therefore, such properties should be refactored into methods
so that callers are not surprised by the unexpectedly poor performance.</p>
<p>This rule tracks calls to the following methods inside properties:</p>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.tolist">Enumerable.ToList</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.toarray">Enumerable.ToArray</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.array.clone">Array.Clone</a> </li>
</ul>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
private List&lt;string&gt; foo = new List&lt;string&gt; { "a", "b", "c" };
private string[] bar = new string[] { "a", "b", "c" };

public IEnumerable&lt;string&gt; Foo =&gt; foo.ToList(); // Noncompliant: collection foo is copied

public IEnumerable&lt;string&gt; Bar =&gt; (string[])bar.Clone(); // Noncompliant: array bar is copied
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
private List&lt;string&gt; foo = new List&lt;string&gt; { "a", "b", "c" };
private string[] bar = new string[] { "a", "b", "c" };

public IEnumerable&lt;string&gt; GetFoo() =&gt; foo.ToList();

public IEnumerable&lt;string&gt; GetBar() =&gt; (string[])bar.Clone();
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/classes-and-structs/properties">Properties (C# Programming
  Guide)</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/classes-and-structs/fields">Fields (C# Programming Guide)</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/concepts/collections">Collections (C#)</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.tolist">Enumerable.ToList</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.toarray">Enumerable.ToArray</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.array.clone">Array.Clone</a> </li>
</ul>

