<p>This rule raises an issue on logging calls inside a <code>catch</code> clause that does not pass the raised <code>Exception</code>.</p>
<h2>Why is this an issue?</h2>
<p>A log entry should contain all the relevant information about the current execution context. The <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.exception">Exception</a> raised in a catch block not only provides the message but also:</p>
<ul>
  <li> the exception type </li>
  <li> the <a href="https://learn.microsoft.com/en-us/dotnet/api/system.exception.stacktrace">stack trace</a> </li>
  <li> any <a href="https://learn.microsoft.com/en-us/dotnet/api/system.exception.innerexception">inner exceptions</a> </li>
  <li> and more about the cause of the error. </li>
</ul>
<p><a href="https://learn.microsoft.com/en-us/dotnet/api/microsoft.extensions.logging.loggerextensions">Logging methods</a> provide overloads that
accept an <code>Exception</code> as a parameter and <a href="https://learn.microsoft.com/en-us/dotnet/core/extensions/logging-providers">logging
providers</a> persist the <code>Exception</code> in a structured way to facilitate the tracking of system failures. Therefore <code>Exceptions</code>
should be passed to the logger.</p>
<p>The rule covers the following logging frameworks:</p>
<ul>
  <li> Nuget package - <a href="https://www.nuget.org/packages/Castle.Core">Castle.Core</a> </li>
  <li> Nuget package - <a href="https://www.nuget.org/packages/Common.Logging.Core">Common.Core</a> </li>
  <li> Nuget package - <a href="https://www.nuget.org/packages/log4net">log4net</a> </li>
  <li> Nuget package - <a href="https://www.nuget.org/packages/NLog">NLog</a> </li>
  <li> Nuget package - <a href="https://www.nuget.org/packages/Microsoft.Extensions.Logging">Microsoft.Extensions.Logging</a> </li>
</ul>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
public bool Save()
{
    try
    {
        DoSave();
        return true;
    }
    catch(IOException)
    {
        logger.LogError("Saving failed.");             // Noncompliant: No specifics about the error are logged
        return false;
    }
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
public bool Save()
{
    try
    {
        DoSave();
        return true;
    }
    catch(IOException exception)
    {
        logger.LogError(exception, "Saving failed.");  // Compliant: Exception details are logged
        return false;
    }
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/core/extensions/logging?tabs=command-line#log-exceptions">Log
  exceptions</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/api/microsoft.extensions.logging.loggerextensions">LoggerExtensions
  Class</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/core/extensions/logging-providers">Logging providers</a> </li>
  <li> Microsoft Learn - <a
  href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/exception-handling-statements#the-try-catch-statement">The
  <code>try-catch</code> statement</a> </li>
  <li> Serilog - <a href="https://github.com/serilog/serilog/wiki/Getting-Started#example-application">Example application</a> </li>
  <li> Serilog Analyzer - <a href="https://github.com/Suchiman/SerilogAnalyzer#serilog001-exception-usage"><code>Serilog001</code>: Exception
  Usage</a> </li>
</ul>

