<p>With Spring, when a request mapping method is configured to accept bean objects as arguments, the framework will automatically bind HTTP parameters
to those objects' properties. If the targeted beans are also persistent entities, the framework will also store those properties in the storage
backend, usually the application’s database.</p>
<h2>Why is this an issue?</h2>
<p>By accepting persistent entities as method arguments, the application allows clients to manipulate the object’s properties directly.</p>
<h3>What is the potential impact?</h3>
<p>Attackers could forge malicious HTTP requests that will alter unexpected properties of persistent objects. This can lead to unauthorized
modifications of the entity’s state. This is known as a <strong>mass assignment</strong> attack.</p>
<p>Depending on the affected objects and properties, the consequences can vary.</p>
<h4>Privilege escalation</h4>
<p>If the affected object is used to store the client’s identity or permissions, the attacker could alter it to change their entitlement on the
application. This can lead to horizontal or vertical privilege escalation.</p>
<h4>Security checks bypass</h4>
<p>Because persistent objects are modified directly without prior logic, attackers could exploit this issue to bypass security measures otherwise
enforced by the application. For example, an attacker might be able to change their e-mail address to an invalid one by directly setting it without
going through the application’s email validation process.</p>
<p>The same could also apply to passwords that attackers could change without complexity validation or knowledge of their current value.</p>
<h2>How to fix it in Java EE</h2>
<h3>Code examples</h3>
<p>The following code is vulnerable to a mass assignment attack because it allows modifying the <code>User</code> persistent entities thanks to
maliciously forged <code>Wish</code> object properties.</p>
<h4>Noncompliant code example</h4>

```java
import javax.persistence.Entity;

@Entity
public class Wish {
  Long productId;
  Long quantity;
  Client client;
}

@Entity
public class Client {
  String clientId;
  String name;
  String password;
}

import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;

@Controller
public class PurchaseOrderController {

  @RequestMapping(path = "/saveForLater", method = RequestMethod.POST)
  public String saveForLater(Wish wish) { // Noncompliant
    session.save(wish);
  }
}
```
<h4>Compliant solution</h4>

```java
public class WishDTO {
  Long productId;
  Long quantity;
  Long clientId;
}

import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;

@Controller
public class PurchaseOrderController {

  @RequestMapping(path = "/saveForLater", method = RequestMethod.POST)
  public String saveForLater(WishDTO wish) {
    Wish persistentWish = new Wish();
    persistentWish.productId = wish.productId
    persistentWish.quantity = wish.quantity
    persistentWish.client = getClientById(with.clientId)
    session.save(persistentWish);
  }
}
```
<h3>How does this work?</h3>
<p>The compliant code implements a Data Transfer Object (DTO) layer. Instead of accepting a persistent <code>Wish</code> entity as a parameter, the
vulnerable method accepts a <code>WishDTO</code> object with a safe, minimal set of properties. It then instantiates a persistent entity and
initializes it based on the DTO properties' values. The resulting object can safely be persisted in the database.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> OWASP - <a href="https://cheatsheetseries.owasp.org/cheatsheets/Mass_Assignment_Cheat_Sheet.html">Mass Assignment Cheat Sheet</a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A08_2021-Software_and_Data_Integrity_Failures/">Top 10 2021 - Category A8 - Software and Data
  Integrity Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A5_2017-Broken_Access_Control">Top 10 2017 - Category A5 - Broken Access
  Control</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/915">CWE-915 - Improperly Controlled Modification of Dynamically-Determined Object
  Attributes</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<p>OWASP O2 Platform Blog - <a href="https://o2platform.files.wordpress.com/2011/07/ounce_springframework_vulnerabilities.pdf">Two Security
Vulnerabilities in the Spring Framework’s MVC</a></p>