<p>This vulnerability allows forging of JSON Web Tokens to impersonate other users.</p>
<h2>Why is this an issue?</h2>
<p>JSON Web Tokens (JWTs), a popular method of securely transmitting information between parties as a JSON object, can become a significant security
risk when they are not properly signed with a robust cipher algorithm, left unsigned altogether, or if the signature is not verified. This
vulnerability class allows malicious actors to craft fraudulent tokens, effectively impersonating user identities. In essence, the integrity of a JWT
hinges on the strength and presence of its signature.</p>
<h3>What is the potential impact?</h3>
<p>When a JSON Web Token is not appropriately signed with a strong cipher algorithm or if the signature is not verified, it becomes a significant
threat to data security and the privacy of user identities.</p>
<h4>Impersonation of users</h4>
<p>JWTs are commonly used to represent user authorization claims. They contain information about the user’s identity, user roles, and access rights.
When these tokens are not securely signed, it allows an attacker to forge them. In essence, a weak or missing signature gives an attacker the power to
craft a token that could impersonate any user. For instance, they could create a token for an administrator account, gaining access to high-level
permissions and sensitive data.</p>
<h4>Unauthorized data access</h4>
<p>When a JWT is not securely signed, it can be tampered with by an attacker, and the integrity of the data it carries cannot be trusted. An attacker
can manipulate the content of the token and grant themselves permissions they should not have, leading to unauthorized data access.</p>
<h2>How to fix it in Java JWT</h2>
<h3>Code examples</h3>
<p>The following code contains examples of JWT encoding and decoding without a strong cipher algorithm.</p>
<h4>Noncompliant code example</h4>

```java
import io.jsonwebtoken.Jwts;

public void encode() {
    Jwts.builder()
        .setSubject(USER_LOGIN)
        .compact(); // Noncompliant
}
```

```java
import io.jsonwebtoken.Jwts;

public void decode() {
    Jwts.parser()
        .setSigningKey(SECRET_KEY)
        .parse(token)
        .getBody(); // Noncompliant
}
```
<h4>Compliant solution</h4>

```java
import io.jsonwebtoken.Jwts;

public void encode() {
    Jwts.builder()
        .setSubject(USER_LOGIN)
        .signWith(SignatureAlgorithm.HS256, SECRET_KEY)
        .compact();
}
```
<p>When using <code>Jwts.parser()</code>, make sure to call <code>parseClaimsJws</code> instead of <code>parse</code> as it throws exceptions for
invalid or missing signatures.</p>

```java
import io.jsonwebtoken.Jwts;

public void decode() {
    Jwts.parser()
        .setSigningKey(SECRET_KEY)
        .parseClaimsJws(token)
        .getBody();
}
```
<h3>How does this work?</h3>
<h4>Always sign your tokens</h4>
<p>The foremost measure to enhance JWT security is to ensure that every JWT you issue is signed. Unsigned tokens are like open books that anyone can
tamper with. Signing your JWTs ensures that any alterations to the tokens after they have been issued can be detected. Most JWT libraries support a
signing function, and using it is usually as simple as providing a secret key when the token is created.</p>
<h4>Choose a strong cipher algorithm</h4>
<p>It is not enough to merely sign your tokens. You need to sign them with a strong cipher algorithm. Algorithms like HS256 (HMAC using SHA-256) are
considered secure for most purposes. But for an additional layer of security, you could use an algorithm like RS256 (RSA Signature with SHA-256),
which uses a private key for signing and a public key for verification. This way, even if someone gains access to the public key, they will not be
able to forge tokens.</p>
<h4>Verify the signature of your tokens</h4>
<p>Resolving a vulnerability concerning the validation of JWT token signatures is mainly about incorporating a critical step into your process:
validating the signature every time a token is decoded. Just having a signed token using a secure algorithm is not enough. If you are not validating
signatures, they are not serving their purpose.</p>
<p>Every time your application receives a JWT, it needs to decode the token to extract the information contained within. It is during this decoding
process that the signature of the JWT should also be checked.</p>
<p>To resolve the issue, follow these instructions:</p>
<ol>
  <li> Use framework-specific functions for signature verification: Most programming frameworks that support JWTs provide specific functions to not
  only decode a token but also validate its signature simultaneously. Make sure to use these functions when handling incoming tokens. </li>
  <li> Handle invalid signatures appropriately: If a JWT’s signature does not validate correctly, it means the token is not trustworthy, indicating
  potential tampering. The action to take when encountering an invalid token should be denying the request carrying it and logging the event for
  further investigation. </li>
  <li> Incorporate signature validation in your tests: When you are writing tests for your application, include tests that check the signature
  validation functionality. This can help you catch any instances where signature verification might be unintentionally skipped or bypassed. </li>
</ol>
<p>By following these practices, you can ensure the security of your application’s JWT handling process, making it resistant to attacks that rely on
tampering with tokens. Validation of the signature needs to be an integral and non-negotiable part of your token handling process.</p>
<h3>Going the extra mile</h3>
<h4>Securely store your secret keys</h4>
<p>Ensure that your secret keys are stored securely. They should not be hard-coded into your application code or checked into your version control
system. Instead, consider using environment variables, secure key management systems, or vault services.</p>
<h4>Rotate your secret keys</h4>
<p>Even with the strongest cipher algorithms, there is a risk that your secret keys may be compromised. Therefore, it is a good practice to
periodically rotate your secret keys. By doing so, you limit the amount of time that an attacker can misuse a stolen key. When you rotate keys, be
sure to allow a grace period where tokens signed with the old key are still accepted to prevent service disruptions.</p>
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">Top 10 2021 Category A2 - Cryptographic Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/347">CWE-347 - Improper Verification of Cryptographic Signature</a> </li>
</ul>