<h2>Why is this an issue?</h2>
<p>An exception thrown inside a promise will not be caught by a nesting <code>try</code> block due to the asynchronous nature of execution.</p>
<p>Promises are designed to propagate errors to the next error handler or catch() block in the promise chain. Promises are asynchronous and operate
outside of the normal call stack. When a Promise is created, it is added to the microtask queue, which is processed after the current call stack has
completed. This means that the try-catch block surrounding the Promise will have already completed by the time the Promise is resolved or rejected.
Therefore, any error occurring within the Promise will not be caught by the try-catch block.</p>

```js
function foo() {
  try { // Noncompliant: Promise rejection will not be caught
    runPromiseThatRejects();
  } catch (e) {
    console.log("Failed to run promise", e);
  }
}
```
<p>Instead of using a try-catch block to handle errors in a Promise chain, use the Promise.catch() method. This method allows you to specify a
callback function that will be executed if the Promise is rejected.</p>

```js
function foo() {
  runPromiseThatRejects().catch(e => console.log("Failed to run promise", e));
}
```
<p>Alternatively, wait for the Promise fulfillment value using <code>await</code>. It is used to unwrap promises and pauses the execution of its
surrounding <code>async</code> function until the promise is settled (that is, fulfilled or rejected). Any errors that occur within the Promise will
be thrown as exceptions.</p>

```js
async function foo() {
  try {
    await runPromiseThatRejects();
  } catch (e) {
    console.log("Failed to run promise", e);
  }
}
```
<p>This rule reports <code>try...catch</code> statements containing nothing else but call(s) to a function returning a <code>Promise</code> (thus,
it’s less likely that <code>catch</code> is intended to catch something else than <code>Promise</code> rejection).</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise">Promise</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/try...catch"><code>try...catch</code></a>
  </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/await"><code>await</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/async_function"><code>async
  function</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/API/HTML_DOM_API/Microtask_guide">Microtasks</a> </li>
</ul>