<h2>Why is this an issue?</h2>
<p>The Kotlin <code>when</code> statement can not only be used to compare a selector against a list of values, but also to select from a list of
conditions.</p>
<p>This is a more readable and elegant alternative to a chain of <code>if</code> statements and should be used instead.</p>
<h3>What is the potential impact?</h3>
<h4>Readability and Understanding</h4>
<p>This change makes it easier to understand a function because it will reduce its complexity. This is because a single <code>when</code> statement
has less cognitive complexity than multiple <code>if</code> statements, even if it has as many cases.</p>
<h2>How to fix it</h2>
<p>Refactor the chain of <code>if</code> statements into a list of conditions for a <code>when</code> statement. Note that the order of the conditions
must stay the same as in the <code>if</code> chain.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```kotlin
fun dispatchFunction(instance: Any) {
    if (instance is Foo) {
        instance.fooFunction()
    } else if (instance is Bar) {
        instance.barFunction()
    } else if (instance is Boo) {
        instance.booFunction()
    } else {
        throw IllegalArgumentException()
    }
}
```
<h4>Compliant solution</h4>

```kotlin
fun dispatchFunction(instance: Any) {
    when (instance) {
        is Foo -> instance.fooFunction()
        is Bar -> instance.barFunction()
        is Boo -> instance.booFunction()
        else -> throw IllegalArgumentException()
    }
}
```
<h4>Noncompliant code example</h4>
<p>When <code>threshold = 2</code></p>

```kotlin
fun compare(a: Int, b: Int): Int {
    // ...
    return if (a > b) { // Noncompliant
        1
    } else if (a < b) {
        -1
    } else {
        0
    }
}
```
<h4>Compliant solution</h4>

```kotlin
fun compare(a: Int, b: Int): Int {
    // ...
    return when { // Compliant
        a > b -> 1
        a < b -> -1
        else -> 0
    }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://kotlinlang.org/docs/control-flow.html#when-expression">Kotlin Docs, When expression</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://www.baeldung.com/kotlin/when">Baeldung, Guide to the “when{}” Block in Kotlin</a> </li>
  <li> <a href="https://superkotlin.com/kotlin-when-statement">Kotlin when: A switch with Superpowers</a> </li>
  <li> <a href="https://www.sonarsource.com/resources/cognitive-complexity">G. Ann Campbell, Cognitive Complexity</a> </li>
</ul>