This check reports a finding when methods, subroutines, or function modules are invoked dynamically using constructs like `CALL METHOD`, `PERFORM ... IN PROGRAM`, or `CALL FUNCTION ...`.

# Why is This a Problem?

Dynamic invocations can introduce several issues in your code:
- **No Compile-Time Validation**: The compiler cannot confirm whether the dynamically referenced entity exists or matches the required signature, increasing the risk of runtime errors.
- **Reduced Readability**: It becomes harder to follow program logic as the target of the call is determined at runtime, making debugging and code comprehension more challenging.
- **Performance Overhead**: Resolving dynamic calls at runtime introduces additional computation compared to static calls.
- **Potential Security Risks**: User-controlled or improperly validated inputs can inadvertently or maliciously affect which entity gets executed, leading to possible vulnerabilities.

By avoiding dynamic invocations, your code becomes easier to understand, safer to execute, and more performant.

# How Can I Resolve This?

Replace dynamic invocations with static references, where the target method, subroutine, or function module is explicitly defined in the code. 

Instead of determining the entity at runtime, consider using design patterns that offer both flexibility and safety. For example, use the **factory pattern** to determine the appropriate object or implementation based on dynamic criteria and call the required logic using static methods.

Where dynamic invocation cannot be avoided (e.g., for specific framework-level scenarios), thoroughly validate and sanitize any inputs that influence the runtime determination of the method, subroutine, or function being called.

# Examples

## Non-Compliant

```abap
CALL METHOD lo_object->('DO_SOMETHING').   " non-compliant: dynamic invocation

DATA(function_name) = 'DO_SOMETHING'.
CALL FUNCTION function_name.               " non-compliant: dynamic invocation

DATA(subroutine_name) = 'DO_SOMETHING'.
PERFORM (subroutine_name).                 " non-compliant: dynamic invocation
```

## Compliant

```abap
lo_object->do_something( ).                " compliant: static invocation

CALL FUNCTION 'DO_SOMETHING'.              " compliant: static invocation

PERFORM do_something.                      " compliant: static invocation
```

# Where Can I Learn More?

- [ABAP Keyword Documentation: CALL METHOD](https://help.sap.com/doc/abapdocu_latest_index_htm/latest/en-US/index.htm?file=abapcall_method_meth_ident_dyna.htm)
- [ABAP Keyword Documentation: CALL FUNCTION](https://help.sap.com/doc/abapdocu_latest_index_htm/latest/en-US/index.htm?file=abapcall_function_general.htm)
- [ABAP Keyword Documentation: PERFORM](https://help.sap.com/doc/abapdocu_latest_index_htm/latest/en-US/index.htm?file=abapperform_form.htm)
