# What Does This Check Look For?

This check searches for the usage of time and date functions defined in `<time.h>` in C and C++ code.
Any functions defined in `<time.h>` should ideally not be used.

# Why is This a Problem?

Using functions defined in `<time.h>` can lead to correctness issues in the code.
These functions may have unspecified, undefined and implementation-defined behavior.
Additionally, they can introduce vulnerabilities related to time-based attacks, such as timing side-channel attacks.
Using these functions can make the code harder to understand, maintain, and test.

# How Can I Resolve This?
Instead of using the time and date functions defined in `<time.h>`, prefer using high-level abstractions provided by the standard library or a third-party library. 
This may be preferable to writing such functions from scratch.
For example, in C++, you can use the `<chrono>` library and its associated classes to perform time and date calculations.
Using higher-level abstractions can ensure deterministic behavior, simplify the code, and make it easier to reason about.

# Examples

## Non-Compliant

```c
#include <time.h>

void doSomething() {
    time_t currentTime = time(NULL); // non-compliant: using time function from <time.h>
    // ...
}
```

## Compliant

```c++
#include <chrono>

void doSomething() {
    auto currentTime = std::chrono::system_clock::now(); // compliant: using std::chrono library
    // ...
}
```

# Where Can I Learn More?

- [C++ Reference: <chrono> library](https://en.cppreference.com/w/cpp/header/chrono)
