Excessive number of variable selection from different objects in a SQL SELECT INTO statement can lead to performance and maintenance challenges due to a large number of unique objects it involves.
Therefore, it is necessary to optimize selection into variables that comes from unique objects for better query efficiency and code readability.

# What Does This Check Look For?

This check identifies instances where variables from a lot of different objects are selected into in a SQL SELECT INTO statement.
It reports a finding when the number of unique objects exceeds a predefined threshold.

# Why is This a Problem?

Selecting numerous columns directly in a SQL query can lead to several issues:

- **Performance Impact:** Retrieving excessive columns might result in slower query execution, especially if unnecessary data is fetched.
- **Readability and Maintenance:** Code becomes less readable and harder to maintain as the number of unique objects involves in the SELECT INTO statement increases. 
Because each unique object implies a separation of context or use-case within the application.

# How Can I Resolve This?

To address this issue, consider alternative approaches to mitigate the impact of selecting many columns in a SQL query:

- **Refactor Queries:** Break down complex queries into smaller, more manageable parts.
- **Consider Cursors and Loops:** When dealing with a substantial number of objects, using cursors and FOR loops might provide a more efficient solution.

# Examples

## Non-Compliant
```plsql
SELECT a, b, c, d
INTO my_obj.a, my_obj.b, my_other_obj.c, my_local_d
FROM ...
```

## Compliant
```plsql

-- Using fewer unique objects
SELECT a, b
INTO my_obj.a, my_obj.b
FROM ...

-- Alternatively, use cursors and loops
DECLARE
  CURSOR c IS SELECT column1, column2, ..., column15 FROM table_name;
  r c%ROWTYPE;
BEGIN
  OPEN c;
  LOOP
    FETCH c INTO r;
    EXIT WHEN c%NOTFOUND;
  -- process record 
  END LOOP;
  CLOSE c;
END;

```