When a `switch` clause is not terminated unconditionally by a statement such as `break` or `exit`, the execution continues in the following clause, which can lead to unexpected behavior.
End all `switch` clauses with a `break` to avoid this.

# Why is This a Problem?

Non-empty `switch` clauses that are not terminated unconditionally by a statement such as `break` or `exit`, or by throwing an exception lead to a fall through into the next clause.
Sometimes this behavior is intended, but it can lead to unexpected execution paths.
Even if the last `case` is the `default` one, a fall through could cause problems in the future when adding new cases after it.

# How Can I Resolve This?

Add a break statement at the end of the `switch` clause.

If the fall through is intended either,
- rewrite the non-compliant `case` to a semantically equivalent `switch` clause which always ends on a `break` statement
or another statement that exits the block (e.g., `return`, `exit`, error throwing), or
- extract helper functions and reuse them across cases as shown below, or
- add the [fallthrough attribute](https://en.cppreference.com/w/cpp/language/attributes/fallthrough) at the end of the clause.

## Exception: Omitting `break` statements in the `default` case

Some coding guidelines do not require to have an exit statement in the `default` case if it is the last `case` of the `switch`.
Usually this would lead to a finding, but this can be tolerated by turning off the respective check option.

# Examples
## Non-Compliant

```cpp
void foo(x) {
    switch(x) {
	case 0:         // compliant
	  break;
	case 1:         // compliant - empty clause
	case 2:         // compliant
	  break;
	case 4:         // non-compliant - fall through
	  a = b;
	case 5:         // non-compliant - break only if a==b
	  if(a == b) {
	    ++a;
	    break;
	  }
	default:        // non-compliant by default (can be toggled via check option)
	  ;
    }  
}
```

Example with several intended fall throughs:
```C
void bar(p) {
  switch (p->state) {
      case ONE: {
          // code for case ONE
          // intentional fall through
      }
      case TWO: {
          // code for case TWO
          // intentional fall through
      }
      case THREE: {
          // code for case THREE
          // intentional fall through
      }
      // ...
  }
}
```

## Compliant

```cpp
void foo(x) {
    switch(x) {
	case 0:
	  break;
	case 1:
	case 2:
	  break;
	case 4:
	  a = b;
	  [[fallthrough]];
	case 5:
	  if(a == b) {
	    ++a;
	  }
	  break;
	default:
	  break;
    }  
}
```

If you are following MISRA, non-empty fall throughs are not allowed at all.
In that case, consider the following refactoring:
```C
void one() {
    // code for case ONE
}

void two() {
    // code for case TWO
}

void three() {
    // code for case THREE
}

void bar(p) {
  switch (p->state) {
      case ONE: {
          one();
          two();
          three();
          break;
      }
      case TWO: {
          two();
          three();
          break;
      }
      case THREE: {
          three();
          break;
      }
      // ...
  }
}
```

# Where Can I Learn More?

- Guideline: MISRA C:2012: Rule 16.3 - An unconditional `break` statement shall terminate every `switch`-clause
- Guideline: AUTOSAR C++14: Rule M6-4-5 - An unconditional `throw` or `break` statement shall terminate every non-empty `switch` clause.
