/*
 * Decompiled with CFR 0.152.
 */
package com.teamscale.index.repository.artifact_store.s3;

import com.teamscale.core.accounts.ExternalCredentials;
import com.teamscale.core.accounts.IExternalCredentialsProvider;
import com.teamscale.core.analysis.configuration.ProjectConfigurationException;
import com.teamscale.index.external.input.external_storage.IOutgoingExternalAnalysisArchive;
import com.teamscale.index.repository.artifact_store.ArtifactStoreItemData;
import com.teamscale.index.repository.artifact_store.ItemQueryResultData;
import com.teamscale.index.repository.artifact_store.SimpleArtifactStoreClientBase;
import com.teamscale.index.repository.artifact_store.s3.S3ArchiveIndex;
import com.teamscale.index.repository.artifact_store.s3.S3RepositoryInfo;
import com.teamscale.index.s3.CredentialsProcess;
import com.teamscale.index.s3.IS3Client;
import com.teamscale.index.s3.IS3ClientFactory;
import com.teamscale.index.s3.S3Exception;
import com.teamscale.index.s3.S3ObjectMetadata;
import com.teamscale.index.s3.S3ObjectSummary;
import com.teamscale.index.s3.S3UriParser;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveInputStream;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Supplier;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.conqat.engine.core.cancel.RescheduleRequestedException;
import org.conqat.engine.index.shared.RepositoryException;
import org.conqat.engine.persistence.store.StorageException;
import org.conqat.engine.resource.util.UniformPathUtils;
import org.conqat.lib.commons.collections.CollectionUtils;
import org.conqat.lib.commons.collections.CounterSet;
import org.conqat.lib.commons.collections.UnmodifiableList;
import org.conqat.lib.commons.date.DateTimeUtils;
import org.conqat.lib.commons.function.BiConsumerWithTwoExceptions;
import org.conqat.lib.commons.string.StringUtils;

public class S3ArtifactStore
extends SimpleArtifactStoreClientBase<S3RepositoryInfo> {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final String ZIP_MIME_TYPE = "application/zip";
    private static final int MAX_ENTRY_MISS_COUNT = 1;
    private static final Duration RESCHEDULE_DELAY = Duration.ofMinutes(1L);
    protected final IS3Client s3;

    protected S3ArtifactStore(IS3Client s3) {
        this.s3 = s3;
    }

    public static @NonNull S3ArtifactStore createS3ArtifactStore(@NonNull ExternalCredentials credentials, @Nullable CredentialsProcess credentialsProcess) throws ProjectConfigurationException {
        IS3ClientFactory clientFactory = IS3ClientFactory.getInstance();
        if (credentialsProcess != null) {
            return new S3ArtifactStore(clientFactory.createWithCredentialsProcess(credentials.uri, credentialsProcess));
        }
        try {
            return new S3ArtifactStore(S3ArtifactStore.createClientWithCredentials(clientFactory, credentials.uri, credentials.username, credentials.password));
        }
        catch (URISyntaxException e) {
            throw new ProjectConfigurationException((Throwable)e);
        }
    }

    @Override
    public void forEachEntryInArchive(String bucketName, String fullPath, BiConsumerWithTwoExceptions<ArchiveInputStream<?>, ArchiveEntry, IOException, StorageException> consumer) throws RepositoryException, StorageException {
        try {
            S3ArtifactStore.forEachEntryInArchiveStream(fullPath, this.s3.getObjectContent(bucketName, fullPath), consumer);
        }
        catch (S3Exception | IOException e) {
            throw new RepositoryException((Throwable)e);
        }
    }

    @Override
    public ItemQueryResultData findItems(S3RepositoryInfo info, int maxCacheAgeSeconds) throws RepositoryException, RescheduleRequestedException {
        try {
            Set<String> keysToScan = S3ArtifactStore.readKeysToScan(info.getArchiveIndex());
            boolean forceFullScan = info.getArchiveIndex().computeWithIncrementalScanLock(S3ArchiveIndex.LockedIncrementalScanIndex::isForceFullScan);
            if (keysToScan.isEmpty() || forceFullScan) {
                return this.findItemsFullScan(info, maxCacheAgeSeconds, forceFullScan);
            }
            return this.findItemsIncrementally(info, keysToScan);
        }
        catch (StorageException e) {
            throw new RepositoryException("There was a problem accessing the storage system", (Throwable)e);
        }
    }

    private static Set<String> readKeysToScan(S3ArchiveIndex index) throws StorageException {
        return index.computeWithIncrementalScanLock(S3ArchiveIndex.LockedIncrementalScanIndex::getKeysToScanIncrementally);
    }

    private ItemQueryResultData findItemsFullScan(S3RepositoryInfo info, long maxCacheAgeSeconds, boolean forceFullScan) throws RepositoryException, StorageException {
        Supplier[] supplierArray = new Supplier[1];
        supplierArray[0] = info::getTargetAsStringForLogging;
        LOGGER.info("Starting full scan for {}.", supplierArray);
        Instant previousLastScan = Instant.ofEpochMilli(info.getArchiveIndex().getLastFullScanTimestamp().orElse(0L));
        Instant now = Instant.now();
        Optional<ItemQueryResultData> oldResults = info.getArchiveIndex().getItemQueryResultData();
        if (!forceFullScan && now.isBefore(previousLastScan.plus(Duration.ofSeconds(maxCacheAgeSeconds))) && oldResults.isPresent()) {
            Supplier[] supplierArray2 = new Supplier[1];
            supplierArray2[0] = info::getTargetAsStringForLogging;
            LOGGER.info("Full scan was skipped; using cached results for {}.", supplierArray2);
            LOGGER.debug("Cached results: \n{}", new Supplier[]{() -> ((ItemQueryResultData)oldResults.get()).stream().map(ArtifactStoreItemData::toString).collect(Collectors.joining("\n"))});
            return oldResults.get();
        }
        S3ArtifactStore.setLastScanTimestamp(now, info, true);
        List<ArtifactStoreItemData> result = this.getAvailableObjectSummaries(info, oldResults.orElse(null)).stream().map(S3ArtifactStore::createItemData).toList();
        ItemQueryResultData itemData = ItemQueryResultData.forFullScanWithAllKeys(info.getKeyIncludeExcludePatterns().filterArtifacts(result));
        info.getArchiveIndex().storeItemQueryResultData(itemData);
        Supplier[] supplierArray3 = new Supplier[2];
        supplierArray3[0] = result::size;
        supplierArray3[1] = info::getTargetAsStringForLogging;
        LOGGER.info("Full scan completed. Found {} items for {}.", supplierArray3);
        LOGGER.debug("Items found: \n{}", new Supplier[]{() -> result.stream().map(ArtifactStoreItemData::toString).collect(Collectors.joining("\n"))});
        return itemData;
    }

    private static @NonNull ArtifactStoreItemData createItemData(S3ObjectSummary objectSummary) {
        String lastModifiedDate = objectSummary.lastModified().toInstant().toString();
        String fullObjectPath = objectSummary.key();
        String itemPath = UniformPathUtils.removeLastSegments((String)fullObjectPath, (int)1);
        String itemName = StringUtils.getLastPart((String)fullObjectPath, (String)"/");
        long contentSize = objectSummary.size();
        return new ArtifactStoreItemData(itemPath, itemName, lastModifiedDate, contentSize);
    }

    private static void setLastScanTimestamp(Instant newScanTimestamp, S3RepositoryInfo info, boolean isFullScan) throws RepositoryException {
        try {
            info.getArchiveIndex().setLastScanTimestamp(newScanTimestamp.toEpochMilli(), isFullScan);
            info.getArchiveIndex().runWithIncrementalScanLock(index -> index.setForceFullScan(false));
        }
        catch (StorageException e) {
            throw new RepositoryException("Couldn't update last scan timestamp.", (Throwable)e);
        }
    }

    private ItemQueryResultData findItemsIncrementally(S3RepositoryInfo info, Set<String> keysToScan) throws RepositoryException, StorageException, RescheduleRequestedException {
        Supplier[] supplierArray = new Supplier[3];
        supplierArray[0] = keysToScan::size;
        supplierArray[1] = info::getTargetAsStringForLogging;
        supplierArray[2] = () -> CollectionUtils.sort((Collection)keysToScan);
        LOGGER.info("Starting incremental scan with {} key(s) for {}: {}", supplierArray);
        LOGGER.debug("Requested keys for incremental scan: \n{}", new Supplier[]{() -> String.join((CharSequence)"\n", keysToScan)});
        S3ArtifactStore.setLastScanTimestamp(DateTimeUtils.now(), info, false);
        IncrementalScanResult scanResult = this.performIncrementalScan(keysToScan, info);
        S3ArtifactStore.storeMissedEntriesAndRescheduleIfNeeded(info.getArchiveIndex(), scanResult.missedEntries(), info);
        return S3ArtifactStore.handleFoundEntries(info, scanResult.foundEntries(), scanResult.data());
    }

    private static void storeMissedEntriesAndRescheduleIfNeeded(S3ArchiveIndex archiveIndex, Set<String> missedEntries, S3RepositoryInfo info) throws StorageException, RescheduleRequestedException {
        archiveIndex.runWithIncrementalScanLock(lockedIndex -> S3ArtifactStore.incrementMissedEntriesAndDiscardEntriesWithTooManyMisses(missedEntries, lockedIndex, info));
        S3ArtifactStore.rescheduleIfMissingEntriesArePresent(missedEntries, info);
    }

    private static void incrementMissedEntriesAndDiscardEntriesWithTooManyMisses(Set<String> missedEntries, S3ArchiveIndex.LockedIncrementalScanIndex lockedIndex, S3RepositoryInfo info) throws StorageException {
        Supplier[] supplierArray = new Supplier[3];
        supplierArray[0] = missedEntries::size;
        supplierArray[1] = info::getTargetAsStringForLogging;
        supplierArray[2] = () -> String.join((CharSequence)"\n", missedEntries);
        LOGGER.info("Missed {} entries for {}: \n{}", supplierArray);
        lockedIndex.incrementAndStoreMissedCounts(missedEntries);
        S3ArtifactStore.removeEntriesWithTooManyMisses(lockedIndex, info);
    }

    private static void rescheduleIfMissingEntriesArePresent(Set<String> missedEntries, S3RepositoryInfo info) throws RescheduleRequestedException {
        if (!missedEntries.isEmpty()) {
            Supplier[] supplierArray = new Supplier[3];
            supplierArray[0] = missedEntries::size;
            supplierArray[1] = info::getTargetAsStringForLogging;
            supplierArray[2] = RESCHEDULE_DELAY::toSeconds;
            LOGGER.warn("Found {} entries that were missed in {}. Rescheduling after {} second(s).", supplierArray);
            throw new RescheduleRequestedException(DateTimeUtils.now().plus(RESCHEDULE_DELAY));
        }
    }

    private static ItemQueryResultData handleFoundEntries(S3RepositoryInfo info, Set<String> foundEntries, List<ArtifactStoreItemData> foundData) throws StorageException {
        info.getArchiveIndex().runWithIncrementalScanLock(archiveIndex -> archiveIndex.removeKeysToScanIncrementally(foundEntries));
        ItemQueryResultData itemData = ItemQueryResultData.forIncrementalAdditiveScanWithAddedKeys(info.getKeyIncludeExcludePatterns().filterArtifacts(foundData));
        ItemQueryResultData fullData = S3ArtifactStore.getOldData(info.getArchiveIndex());
        fullData.results.addAll(itemData.results);
        info.getArchiveIndex().storeItemQueryResultData(fullData);
        return itemData;
    }

    private static ItemQueryResultData getOldData(S3ArchiveIndex archiveIndex) throws StorageException {
        return archiveIndex.getItemQueryResultData().orElse(ItemQueryResultData.emptyResultData());
    }

    private static void removeEntriesWithTooManyMisses(S3ArchiveIndex.LockedIncrementalScanIndex index, S3RepositoryInfo info) throws StorageException {
        CounterSet<String> countedMisses = index.getMissedEntryCounts();
        Set entriesWithTooManyMisses = countedMisses.getKeysWithCountAbove(1);
        if (!entriesWithTooManyMisses.isEmpty()) {
            Supplier[] supplierArray = new Supplier[4];
            supplierArray[0] = entriesWithTooManyMisses::size;
            supplierArray[1] = info::getTargetAsStringForLogging;
            supplierArray[2] = () -> 1;
            supplierArray[3] = () -> String.join((CharSequence)"\n", entriesWithTooManyMisses);
            LOGGER.warn("Removing {} key(s) for {} because they were not found {} times: \n{}", supplierArray);
            index.removeKeysToScanIncrementally(entriesWithTooManyMisses);
        }
    }

    private IncrementalScanResult performIncrementalScan(Set<String> keysToScan, S3RepositoryInfo info) {
        ArrayList<ArtifactStoreItemData> result = new ArrayList<ArtifactStoreItemData>();
        HashSet<String> foundEntries = new HashSet<String>();
        for (String s3Key : keysToScan) {
            this.retrieveKeyMetaData(s3Key, info).ifPresent(data -> {
                result.add((ArtifactStoreItemData)data);
                foundEntries.add(s3Key);
            });
        }
        return new IncrementalScanResult(result, foundEntries, keysToScan.stream().filter(Predicate.not(foundEntries::contains)).collect(Collectors.toSet()));
    }

    private Optional<ArtifactStoreItemData> retrieveKeyMetaData(String s3Key, S3RepositoryInfo info) {
        try {
            S3ObjectMetadata objectMetadata = this.s3.getObjectMetadata(info.getRepositoryOrBucketName(), s3Key);
            String lastModifiedDate = objectMetadata.getLastModified().toInstant().toString();
            String itemPath = UniformPathUtils.removeLastSegments((String)s3Key, (int)1);
            String itemName = StringUtils.getLastPart((String)s3Key, (String)"/");
            long contentSize = objectMetadata.getContentLength();
            return Optional.of(new ArtifactStoreItemData(itemPath, itemName, lastModifiedDate, contentSize));
        }
        catch (S3Exception e) {
            if (e.isNotFoundError()) {
                Supplier[] supplierArray = new Supplier[2];
                supplierArray[0] = () -> s3Key;
                supplierArray[1] = info::getTargetAsStringForLogging;
                LOGGER.info("Ignoring key '{}' for incremental scan of {}, as it was removed from S3. The removal will be taken into account during the next full scan.", supplierArray);
            } else {
                Supplier[] supplierArray = new Supplier[2];
                supplierArray[0] = () -> s3Key;
                supplierArray[1] = info::getTargetAsStringForLogging;
                LOGGER.atError().withThrowable((Throwable)e).log("Couldn't retrieve metadata for key '{}' from S3 {} during incremental scan. Continuing incremental retrieval, the missing key will be picked up during the next full scan.", supplierArray);
            }
            return Optional.empty();
        }
    }

    @Override
    public void putArchive(String bucketName, IOutgoingExternalAnalysisArchive archive) throws RepositoryException {
        LOGGER.traceEntry("Uploading archive '{}' with {} byte(s) to bucket '{}'.", new Object[]{archive.getTargetPath(), archive.getDataSize(), bucketName});
        try {
            this.s3.putObject(bucketName, archive.getTargetPath(), archive.getDataStream(), new S3ObjectMetadata(ZIP_MIME_TYPE, archive.getDataSize()));
            LOGGER.traceExit("Successfully uploaded archive '{}' to bucket '" + bucketName + "'.", (Object)archive.getTargetPath());
        }
        catch (S3Exception e) {
            throw (RepositoryException)LOGGER.throwing(Level.TRACE, (Throwable)new RepositoryException((Throwable)e));
        }
    }

    @Override
    public void deleteArchive(String repositoryOrBucketName, String path) throws RepositoryException {
        LOGGER.traceEntry(repositoryOrBucketName, new Object[]{path});
        try {
            this.s3.deleteObject(repositoryOrBucketName, path);
        }
        catch (S3Exception e) {
            throw (RepositoryException)LOGGER.throwing(Level.TRACE, (Throwable)new RepositoryException((Throwable)e));
        }
        LOGGER.traceExit((Object)"Bucket '%s': deleted archive '%s'.".formatted(repositoryOrBucketName, path));
    }

    @Override
    public Set<String> fetchBranches(S3RepositoryInfo info) throws RepositoryException, ProjectConfigurationException {
        Pattern compiledBranchExtractionPattern = Pattern.compile(info.getBranchExtractionPattern());
        HashSet<String> uniqueBranches = new HashSet<String>();
        UnmodifiableList<S3ObjectSummary> availableObjectSummaries = this.getAvailableObjectSummaries(info, null);
        for (S3ObjectSummary objectSummary : availableObjectSummaries) {
            Matcher matcher = compiledBranchExtractionPattern.matcher(objectSummary.key());
            if (!matcher.matches() || matcher.groupCount() <= 0) continue;
            uniqueBranches.add(matcher.group(1));
        }
        return uniqueBranches;
    }

    @Override
    public void testConnection(String repositoryOrBucketName, String uploadPathPrefix) throws ProjectConfigurationException {
        try {
            String testUploadKey = StringUtils.ensureEndsWith((String)uploadPathPrefix.trim(), (String)"/") + "test_write_permission";
            this.s3.putObject(repositoryOrBucketName, testUploadKey, "");
            LOGGER.trace("Connection test with bucket '{}' and test upload key '{}' was successful.", (Object)repositoryOrBucketName, (Object)testUploadKey);
            this.s3.deleteObject(repositoryOrBucketName, testUploadKey);
        }
        catch (S3Exception exception) {
            throw new ProjectConfigurationException((Throwable)exception);
        }
    }

    private UnmodifiableList<S3ObjectSummary> getAvailableObjectSummaries(S3RepositoryInfo info, @Nullable ItemQueryResultData previousResults) throws RepositoryException {
        List<String> keyPrefixes = info.getKeyPrefixes();
        if (keyPrefixes.isEmpty()) {
            keyPrefixes = List.of("");
        }
        return this.getAvailableObjectSummariesForKeyPrefixes(info, keyPrefixes, previousResults);
    }

    private @NonNull UnmodifiableList<S3ObjectSummary> getAvailableObjectSummariesForKeyPrefixes(S3RepositoryInfo info, List<String> keyPrefixes, @Nullable ItemQueryResultData previousResults) throws RepositoryException {
        try {
            ArrayList<S3ObjectSummary> list = new ArrayList<S3ObjectSummary>();
            for (String prefix : keyPrefixes) {
                if (previousResults != null) {
                    list.addAll(this.s3.getAllObjectSummariesOptimizedByKnownPaths(info.getRepositoryOrBucketName(), prefix, previousResults));
                    continue;
                }
                list.addAll(this.s3.getAllObjectSummaries(info.getRepositoryOrBucketName(), prefix));
            }
            return CollectionUtils.asUnmodifiable(list);
        }
        catch (S3Exception e) {
            throw new RepositoryException((Throwable)e);
        }
    }

    public void validateBucket(String bucketName, List<String> keyPrefixes) throws S3Exception {
        if (keyPrefixes.isEmpty()) {
            this.s3.validateRead(bucketName, "");
        } else {
            for (String keyPrefix : keyPrefixes) {
                this.s3.validateRead(bucketName, keyPrefix.trim());
            }
        }
    }

    public void validateCredentials() throws S3Exception {
        this.s3.validateCredentials();
    }

    protected static IS3Client createClientWithCredentials(@NonNull IS3ClientFactory clientFactory, @NonNull String baseUri, @NonNull String username, @NonNull String password) throws URISyntaxException {
        IExternalCredentialsProvider credentialsProvider = null;
        if (!StringUtils.isEmpty((String)username) && !StringUtils.isEmpty((String)password)) {
            credentialsProvider = credentialsName -> new ExternalCredentials(credentialsName, baseUri, username, password);
        }
        try {
            return clientFactory.createForUri(credentialsProvider, S3UriParser.parseUri(new URI(baseUri)));
        }
        catch (StorageException e) {
            throw new AssertionError("This should not be possible, as we pass a credential provider without storage access!", e);
        }
    }

    private record IncrementalScanResult(List<ArtifactStoreItemData> data, Set<String> foundEntries, Set<String> missedEntries) {
    }
}

