<h2>Why is this an issue?</h2>
<p>Returning <code>null</code> from a non-<code>async</code> <code>Task</code>/<code>Task&lt;TResult&gt;</code> method will cause a
<code>NullReferenceException</code> at runtime if the method is awaited. This problem can be avoided by returning <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.completedtask"><code>Task.CompletedTask</code></a> or <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.fromresult"><code>Task.FromResult&lt;TResult&gt;(null)</code></a>
respectively.</p>
<pre>
public Task DoFooAsync()
{
    return null;               // Noncompliant: Causes a NullReferenceException if awaited.
}

public async Task Main()
{
    await DoFooAsync();        // NullReferenceException
}
</pre>
<h2>How to fix it</h2>
<p>Instead of <code>null</code> <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.completedtask"><code>Task.CompletedTask</code></a> or <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.fromresult"><code>Task.FromResult&lt;TResult&gt;(null)</code></a>
should be returned.</p>
<h3>Code examples</h3>
<p>A <code>Task</code> returning method can be fixed like so:</p>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
public Task DoFooAsync()
{
    return null;               // Noncompliant: Causes a NullReferenceException if awaited.
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
public Task DoFooAsync()
{
    return Task.CompletedTask; // Compliant: Method can be awaited.
}
</pre>
<p>A <code>Task&lt;TResult&gt;</code> returning method can be fixed like so:</p>
<h4>Noncompliant code example</h4>
<pre data-diff-id="2" data-diff-type="noncompliant">
public Task&lt;object&gt; GetFooAsync()
{
    return null;                          // Noncompliant: Causes a NullReferenceException if awaited.
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="2" data-diff-type="compliant">
public Task&lt;object&gt; GetFooAsync()
{
    return Task.FromResult&lt;object&gt;(null); // Compliant: Method can be awaited.
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Microsoft Learn - <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.completedtask"><code>Task.CompletedTask</code> Property</a> </li>
  <li> Microsoft Learn - <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.fromresult"><code>Task.FromResult&lt;TResult&gt;(TResult)</code>
  Method</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> StackOverflow - Answer by Stephen Cleary for <a href="https://stackoverflow.com/a/45350108">Is it better to return an empty task or null?</a>
  </li>
  <li> StackOverflow - Answer by Stephen Cleary for <a href="https://stackoverflow.com/a/27551261">Best way to handle null task inside async
  method?</a> </li>
  <li> C# Language Design - <a href="https://github.com/dotnet/csharplang/issues/35">Proposal Champion "Null-conditional await"</a> </li>
</ul>

