<h2>Why is this an issue?</h2>
<p>Both the <code>List.TrueForAll</code> method and the <code>IEnumerable.All</code> method can be used to check if all list elements satisfy a given
condition in a collection. However, <code>List.TrueForAll</code> can be faster than <code>IEnumerable.All</code> for <code>List</code> objects. The
performance difference may be minor for small collections, but for large collections, it can be noticeable.</p>
<p><strong>Applies to</strong></p>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.list-1.trueforall">List</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.array.trueforall">Array</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.trueforall">ImmutableList</a> </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.builder.trueforall">ImmutableList.Builder</a> </li>
</ul>
<h3>What is the potential impact?</h3>
<p>We measured at least 4x improvement both in execution time. For more details see the <code>Benchmarks</code> section from the <code>More
info</code> tab.</p>
<h2>How to fix it</h2>
<p>The <code>TrueForAll</code> method is defined on the collection class, and it has the same signature as the <code>All</code> extension method. The
method can be replaced in place.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
public bool AreAllEven(List&lt;int&gt; data) =&gt;
    data.All(x =&gt; x % 2 == 0);
</pre>
<pre data-diff-id="2" data-diff-type="noncompliant">
public bool AreAllEven(int[] data) =&gt;
    data.All(x =&gt; x % 2 == 0);
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
public bool AreAllEven(List&lt;int&gt; data) =&gt;
    data.TrueForAll(x =&gt; x % 2 == 0);
</pre>
<pre data-diff-id="2" data-diff-type="compliant">
public bool AreAllEven(int[] data) =&gt;
    Array.TrueForAll(data, x =&gt; x % 2 == 0);
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.list-1.trueforall">List&lt;T&gt;.TrueForAll(Predicate&lt;T&gt;)</a>
  </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.array.trueforall">Array.TrueForAll&lt;T&gt;(T[], Predicate&lt;T&gt;)</a> </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.trueforall">ImmutableList&lt;T&gt;.TrueForAll(Predicate&lt;T&gt;)</a> </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.builder.trueforall">ImmutableList&lt;T&gt;.Builder.TrueForAll(Predicate&lt;T&gt;)</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.all">Enumerable.All&lt;TSource&gt;</a> </li>
</ul>
<h3>Benchmarks</h3>
<table>
  <colgroup>
    <col style="width: 20%;">
    <col style="width: 20%;">
    <col style="width: 20%;">
    <col style="width: 20%;">
    <col style="width: 20%;">
  </colgroup>
  <thead>
    <tr>
      <th>Method</th>
      <th>Runtime</th>
      <th>Mean</th>
      <th>Standard Deviation</th>
      <th>Allocated</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <td><p>TrueForAll</p></td>
      <td><p>.NET 7.0</p></td>
      <td><p>1.302 ms</p></td>
      <td><p>0.0027 ms</p></td>
      <td><p>1 B</p></td>
    </tr>
    <tr>
      <td><p>All</p></td>
      <td><p>.NET 7.0</p></td>
      <td><p>6.279 ms</p></td>
      <td><p>0.0181 ms</p></td>
      <td><p>40004 B</p></td>
    </tr>
    <tr>
      <td><p>TrueForAll</p></td>
      <td><p>.NET Framework 4.6.2</p></td>
      <td><p>1.105 ms</p></td>
      <td><p>0.0142 ms</p></td>
      <td><p>-</p></td>
    </tr>
    <tr>
      <td><p>All</p></td>
      <td><p>.NET Framework 4.6.2</p></td>
      <td><p>4.968 ms</p></td>
      <td><p>0.0143 ms</p></td>
      <td><p>40128 B</p></td>
    </tr>
  </tbody>
</table>
<h4>Glossary</h4>
<ul>
  <li> <a href="https://en.wikipedia.org/wiki/Arithmetic_mean">Mean</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Standard_deviation">Standard Deviation</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Memory_management">Allocated</a> </li>
</ul>
<p>The results were generated by running the following snippet with <a href="https://github.com/dotnet/BenchmarkDotNet">BenchmarkDotNet</a>:</p>
<pre>
private List&lt;int&gt; data;

[Params(10_000)]
public int N { get; set; }

[GlobalSetup]
public void Setup() =&gt;
    data = Enumerable.Range(0, N).Select(x =&gt; 42).ToList();

[Benchmark]
public void TrueForAll()
{
    for (var i = 0; i &lt; N; i++)
    {
        _ = data.TrueForAll(x =&gt; x == 42);  // List&lt;T&gt;.TrueForAll
    }
}

[Benchmark(Baseline = true)]
public void All()
{
    for (var i = 0; i &lt; N; i++)
    {
        _ = data.All(x =&gt; x == 42);         // Enumerable.All&lt;TSource&gt;
    }
}
</pre>
<p>Hardware configuration:</p>
<pre>
BenchmarkDotNet=v0.13.5, OS=Windows 10 (10.0.19045.2846/22H2/2022Update)
12th Gen Intel Core i7-12800H, 1 CPU, 20 logical and 14 physical cores
.NET SDK=7.0.203
  [Host]               : .NET 7.0.5 (7.0.523.17405), X64 RyuJIT AVX2
  .NET 7.0             : .NET 7.0.5 (7.0.523.17405), X64 RyuJIT AVX2
  .NET Framework 4.6.2 : .NET Framework 4.8 (4.8.4614.0), X64 RyuJIT VectorSize=256
</pre>

