<p>A code block should not contain too many logging statements of a specific level.</p>
<h2>Why is this an issue?</h2>
<p>Excessive logging within a code block can lead to several problems:</p>
<ul>
  <li> <strong>Log file overload</strong>: generating an overwhelming number of log entries can fill up disk space quickly (thus increasing the
  storage space cost) and make it challenging to identify important log events promptly. </li>
  <li> <strong>Performance degradation</strong>: writing a large number of log statements can impact the performance of an application, especially
  when the logs are placed in frequently executed paths. </li>
  <li> <strong>Code readability and maintainability</strong>: excessive logging can clutter the code and increase the code’s complexity, making it
  difficult for developers to identify essential logic. </li>
</ul>
<p>Only the logging statements that are directly within the <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/tour-of-csharp/program-building-blocks#statements">code block</a> will be counted, and any
logging statements within nested blocks will count towards their own. For example consider the snippet below:</p>
<pre>
void MyMethod(List&lt;MyObject&gt; items)
{
    logger.Debug("The operation started");
    foreach(var item in items)
    {
        logger.Debug($"Evaluating {item.Name}");
        var result = Evaluate(item);
        logger.Debug($"Evaluating resulted in {result}");
    }
    logger.Debug("The operation ended");
}
</pre>
<p>The rule will count 2 logging statements that are within the method block (namely <code>logger.Debug("The operation started")</code> and
<code>logger.Debug("The operation ended")</code>). Any statements within nested blocks, such as the <code>foreach</code> block will be counted
separately. The rule considers the log level of the calls, as follows:</p>
<ul>
  <li> <strong>Debug</strong>, <strong>Trace</strong> and <strong>Verbose</strong> logging level statements will count together and raise when the
  <strong><em>Debug threshold</em></strong> parameter is exceeded (default value: <em>4</em>); </li>
  <li> <strong>Information</strong> logging level statements will raise when the <strong><em>Information threshold</em></strong> parameter is exceeded
  (default value: <em>2</em>); </li>
  <li> <strong>Warning</strong> logging level statements will raise when the <strong><em>Warning threshold</em></strong> parameter is exceeded
  (default value: <em>1</em>); </li>
  <li> <strong>Error</strong> and <strong>Fatal</strong> logging level statements will count together and raise when the <strong><em>Error
  threshold</em></strong> parameter is exceeded (default value: <em>1</em>); </li>
</ul>
<p>The most popular logging frameworks are supported:</p>
<ul>
  <li> Nuget package - <a href="https://www.nuget.org/packages/Microsoft.Extensions.Logging">Microsoft.Extensions.Logging</a> </li>
  <li> Nuget package - <a href="https://www.nuget.org/packages/Serilog">Serilog</a> </li>
  <li> Nuget package - <a href="https://www.nuget.org/packages/Castle.Core">Castle.Core</a> </li>
  <li> Nuget package - <a href="https://www.nuget.org/packages/NLog">NLog</a> </li>
  <li> Nuget package - <a href="https://www.nuget.org/packages/log4net">log4net</a> </li>
</ul>
<h2>How to fix it</h2>
<p>Reduce the number of specific logging level calls within the code block by identifying and selecting essential log statements with relevant
information, necessary for understanding the flow of execution or diagnosing issues.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>With the default Information threshold parameter value 2:</p>
<pre data-diff-id="1" data-diff-type="noncompliant">
void MyMethod(List&lt;MyObject&gt; items)
{
    logger.Debug("The operation started");
    foreach(var item in items)
    {
        logger.Information($"Evaluating {item.Name}"); // Noncompliant
        var result = Evaluate(item);
        logger.Information($"Evaluating resulted in {result}"); // Secondary 1
        if (item.Name is string.Empty)
        {
            logger.Error("Invalid item name");
        }
        logger.Information("End item evaluation"); // Secondary 2
    }
    logger.Debug("The operation ended");
}
</pre>
<h4>Compliant solution</h4>
<p>With the default Information threshold parameter value 2:</p>
<pre data-diff-id="1" data-diff-type="compliant">
void MyMethod(List&lt;MyObject&gt; items)
{
    logger.Debug("The operation started");
    foreach(var item in items)
    {
        logger.Information($"Evaluating {item.Name}");
        var result = Evaluate(item);
        if (item.Name is string.Empty)
        {
            logger.Error("Invalid item name");
        }
        logger.Information($"End item evaluation with result: {result}");
    }
    logger.Debug("The operation ended");
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/tour-of-csharp/program-building-blocks#statements">Code blocks</a>
  </li>
  <li> Microsoft Learn - <a
  href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/exception-handling-statements">Exception-handling statements</a>
  </li>
</ul>

