﻿## SA1200

<table>
<tr>
  <td>TypeName</td>
  <td>SA1200UsingDirectivesMustBePlacedCorrectly</td>
</tr>
<tr>
  <td>CheckId</td>
  <td>SA1200</td>
</tr>
<tr>
  <td>Category</td>
  <td>Ordering Rules</td>
</tr>
</table>

## Cause

A C# using directive is placed outside of a namespace element.

> :memo: The behavior of this rule can change based on the configuration of the `usingDirectivesPlacement` property in **stylecop.json**. See [Configuration.md](Configuration.md#using-directives) for more information.

## Rule description

A violation of this rule occurs when a using directive or a using-alias directive is placed outside of a namespace element, unless the file does not contain any namespace elements.

For example, the following code would result in two violations of this rule.

```csharp
using System;
using Guid = System.Guid;

namespace Microsoft.Sample
{
    public class Program
    {
    }
}
```

The following code, however, would not result in any violations of this rule:

```csharp
namespace Microsoft.Sample
{
    using System;
    using Guid = System.Guid;
 
    public class Program
    {
    }
}
```

There are subtle differences between placing using directives within a namespace element, rather than outside of the namespace, including:

1. Placing using-alias directives within the namespace eliminates compiler confusion between conflicting types.

2. When multiple namespaces are defined within a single file, placing using directives within the namespace elements scopes references and aliases.

### 1. Eliminating Type Confusion

Consider the following code, which contains a using-alias directive defined outside of the namespace element. The code creates a new class called *Guid*, and also defines a using-alias directive to map the name *Guid* to the type *System.Guid*. Finally, the code creates an instance of the type *Guid*:

```csharp
using Guid = System.Guid;
 
namespace Microsoft.Sample
{
    public class Guid
    {
        public Guid(string s)
        {
        }
    }

    public class Program
    {
        public static void Main(string[] args)
        {
            Guid g = new Guid("hello");
        }
    }
}
```

This code will compile cleanly, without any compiler errors. However, it is unclear which version of the *Guid* type is being allocated. If the using directive is moved inside of the namespace, as shown below, a compiler error will occur:

```csharp
namespace Microsoft.Sample
{
    using Guid = System.Guid;

    public class Guid
    {
        public Guid(string s)
        {
        }
    }

    public class Program
    {
        public static void Main(string[] args)
        {
            Guid g = new Guid("hello");
        }
    }
}
```

The code fails on the following compiler error, found on the line containing `Guid g = new Guid("hello");`

> CS0576: Namespace 'Microsoft.Sample' contains a definition conflicting with alias 'Guid'

The code creates an alias to the *System.Guid* type called *Guid*, and also creates its own type called *Guid* with a matching constructor interface. Later, the code creates an instance of the type *Guid*. To create this instance, the compiler must choose between the two different definitions of *Guid*. When the using-alias directive is placed outside of the namespace element, the compiler will choose the local definition of *Guid* defined within the local namespace, and completely ignore the using-alias directive defined outside of the namespace. This, unfortunately, is not obvious when reading the code.

When the using-alias directive is positioned within the namespace, however, the compiler has to choose between two different, conflicting *Guid* types both defined within the same namespace. Both of these types provide a matching constructor. The compiler is unable to make a decision, so it flags the compiler error.

Placing the using-alias directive outside of the namespace is a bad practice because it can lead to confusion in situations such as this, where it is not obvious which version of the type is actually being used. This can potentially lead to a bug which might be difficult to diagnose.

Placing using-alias directives within the namespace element eliminates this as a source of bugs. 

### 2. Multiple Namespaces

Placing multiple namespace elements within a single file is generally a bad idea, but if and when this is done, it is a good idea to place all using directives within each of the namespace elements, rather than globally at the top of the file. This will scope the namespaces tightly, and will also help to avoid the kind of behavior described above.

It is important to note that when code has been written with using directives placed outside of the namespace, care should be taken when moving these directives within the namespace, to ensure that this is not changing the semantics of the code. As explained above, placing using-alias directives within the namespace element allows the compiler to choose between conflicting types in ways that will not happen when the directives are placed outside of the namespace.

## How to fix violations

To fix a violation of this rule, move all using directives and using-alias directives within the namespace element.

## How to suppress violations

```csharp
[SuppressMessage("StyleCop.CSharp.OrderingRules", "SA1200:UsingDirectivesMustBePlacedWithinNamespace", Justification = "Reviewed.")]
```

```csharp
#pragma warning disable SA1200 // UsingDirectivesMustBePlacedWithinNamespace
#pragma warning restore SA1200 // UsingDirectivesMustBePlacedWithinNamespace
```
