<h2>Why is this an issue?</h2>
<p>Connections, streams, files, and other classes that implement the <code>Closeable</code> interface or its super-interface,
<code>AutoCloseable</code>, needs to be closed after use. Further, that <code>close</code> call must be made in a <code>finally</code> block otherwise
an exception could keep the call from being made. Preferably, when class implements <code>AutoCloseable</code>, resource should be created using
"try-with-resources" pattern and will be closed automatically.</p>
<p>Failure to properly close resources will result in a resource leak which could bring first the application and then perhaps the box the application
is on to their knees.</p>
<h3>Noncompliant code example</h3>

```java
private void readTheFile() throws IOException {
  Path path = Paths.get(this.fileName);
  BufferedReader reader = Files.newBufferedReader(path, this.charset);
  // ...
  reader.close();  // Noncompliant
  // ...
  Files.lines("input.txt").forEach(System.out::println); // Noncompliant: The stream needs to be closed
}

private void doSomething() {
  OutputStream stream = null;
  try {
    for (String property : propertyList) {
      stream = new FileOutputStream("myfile.txt");  // Noncompliant
      // ...
    }
  } catch (Exception e) {
    // ...
  } finally {
    stream.close();  // Multiple streams were opened. Only the last is closed.
  }
}
```
<h3>Compliant solution</h3>

```java
private void readTheFile(String fileName) throws IOException {
    Path path = Paths.get(fileName);
    try (BufferedReader reader = Files.newBufferedReader(path, StandardCharsets.UTF_8)) {
      reader.readLine();
      // ...
    }
    // ..
    try (Stream<String> input = Files.lines("input.txt"))  {
      input.forEach(System.out::println);
    }
}

private void doSomething() {
  OutputStream stream = null;
  try {
    stream = new FileOutputStream("myfile.txt");
    for (String property : propertyList) {
      // ...
    }
  } catch (Exception e) {
    // ...
  } finally {
    stream.close();
  }
}
```
<h3>Exceptions</h3>
<p>Instances of the following classes are ignored by this rule because <code>close</code> has no effect:</p>
<ul>
  <li> <code>java.io.ByteArrayOutputStream</code> </li>
  <li> <code>java.io.ByteArrayInputStream</code> </li>
  <li> <code>java.io.CharArrayReader</code> </li>
  <li> <code>java.io.CharArrayWriter</code> </li>
  <li> <code>java.io.StringReader</code> </li>
  <li> <code>java.io.StringWriter</code> </li>
</ul>
<p>Java 7 introduced the try-with-resources statement, which implicitly closes <code>Closeables</code>. All resources opened in a try-with-resources
statement are ignored by this rule.</p>

```java
try (BufferedReader br = new BufferedReader(new FileReader(fileName))) {
  //...
}
catch ( ... ) {
  //...
}
```
<h2>Resources</h2>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/459">CWE-459 - Incomplete Cleanup</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/772">CWE-772 - Missing Release of Resource after Effective Lifetime</a> </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/vjdGBQ">CERT, FIO04-J.</a> - Release resources when they are no longer needed </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/QtUxBQ">CERT, FIO42-C.</a> - Close files when they are no longer needed </li>
  <li> <a href="https://docs.oracle.com/javase/tutorial/essential/exceptions/tryResourceClose.html">Try With Resources</a> </li>
</ul>