<h2>Why is this an issue?</h2>
<p>Transactional methods have a propagation type parameter in the @Transaction annotation that specifies the requirements about the transactional
context in which the method can be called and how it creates, appends, or suspends an ongoing transaction.</p>
<p>When an instance that contains transactional methods is injected, Spring uses proxy objects to wrap these methods with the actual transaction
code.</p>
<p>However, if a transactional method is called from another method in the same class, the <code>this</code> argument is used as the receiver instance
instead of the injected proxy object, which bypasses the wrapper code. This results in specific transitions from one transactional method to another,
which are not allowed:</p>
<table>
  <colgroup>
    <col style="width: 50%;">
    <col style="width: 50%;">
  </colgroup>
  <thead>
    <tr>
      <th>From</th>
      <th>To</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <td><p>non-<code>@Transactional</code></p></td>
      <td><p>MANDATORY, NESTED, REQUIRED, REQUIRES_NEW</p></td>
    </tr>
    <tr>
      <td><p>MANDATORY</p></td>
      <td><p>NESTED, NEVER, NOT_SUPPORTED, REQUIRES_NEW</p></td>
    </tr>
    <tr>
      <td><p>NESTED</p></td>
      <td><p>NESTED, NEVER, NOT_SUPPORTED, REQUIRES_NEW</p></td>
    </tr>
    <tr>
      <td><p>NEVER</p></td>
      <td><p>MANDATORY, NESTED, REQUIRED, REQUIRES_NEW</p></td>
    </tr>
    <tr>
      <td><p>NOT_SUPPORTED</p></td>
      <td><p>MANDATORY, NESTED, REQUIRED, REQUIRES_NEW</p></td>
    </tr>
    <tr>
      <td><p>REQUIRED or <code>@Transactional</code></p></td>
      <td><p>NESTED, NEVER, NOT_SUPPORTED, REQUIRES_NEW</p></td>
    </tr>
    <tr>
      <td><p>REQUIRES_NEW</p></td>
      <td><p>NESTED, NEVER, NOT_SUPPORTED, REQUIRES_NEW</p></td>
    </tr>
    <tr>
      <td><p>SUPPORTS</p></td>
      <td><p>MANDATORY, NESTED, NEVER, NOT_SUPPORTED, REQUIRED, REQUIRES_NEW</p></td>
    </tr>
  </tbody>
</table>
<h2>How to fix it</h2>
<p>Change the corresponding functions into a compatible propagation type.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
public void doTheThing() {
  // ...
  actuallyDoTheThing(); // Noncompliant, call from non-transactional to transactional
}

@Transactional
public void actuallyDoTheThing() {
  // ...
}
```
<h4>Compliant solution</h4>

```java
@Transactional
public void doTheThing() {
  // ...
  actuallyDoTheThing(); // Compliant
}

@Transactional
public void actuallyDoTheThing() {
  // ...
}
```
<h4>Noncompliant code example</h4>

```java
@Transactional
public void doTheThing() {
  // ...
  actuallyDoTheThing(); // Noncompliant, call from REQUIRED to REQUIRES_NEW
}

@Transactional(propagation = Propagation.REQUIRES_NEW)
public void actuallyDoTheThing() {
  // ...
}
```
<h4>Compliant solution</h4>

```java
@Transactional
public void doTheThing() {
  // ...
  actuallyDoTheThing(); // Compliant, call from REQUIRED to MANDATORY
}

@Transactional(propagation = Propagation.MANDATORY)
public void actuallyDoTheThing() {
  // ...
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/transaction/annotation/Propagation.html">Spring
  Framework 6 API: Enum Class propagation</a> </li>
  <li> <a href="https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/transaction/annotation/Transactional.html">Spring
  Framework 6 API: Annotation Interface Transactional</a> </li>
  <li> <a href="https://docs.spring.io/spring-framework/reference/data-access/transaction/declarative/tx-propagation.html">Spring 6 Documentation:
  Transaction Propagation</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://www.baeldung.com/spring-transactional-propagation-isolation">Baeldung: Transaction Propagation and Isolation in Spring
  @Transactional</a> </li>
  <li> <a href="https://dzone.com/articles/spring-transaction-propagation">DZone: Spring Transaction Propagation in a Nutshell</a> </li>
</ul>