<p>The <em>Frame Rate API</em> allows applications to communicate their desired frame rate to the <em>Android platform</em> to enhance the user
experience. The API is useful since many devices now offer varying refresh rates like 60Hz, 90Hz, or 120Hz.</p>
<h2>Why is this an issue?</h2>
<p>Standard applications don’t require a display refresh rate above 60Hz, hence it is advisable to avoid higher frequencies to avoid unnecessary
energy consumption.</p>
<p>The rule flags an issue when <code>setFrameRate()</code> is invoked with a frameRate higher than 60Hz for <code>android.view.Surface</code> and
<code>android.view.SurfaceControl.Transaction</code>.</p>
<p>It’s important to note that the scheduler considers several factors when determining the display refresh rate. Therefore, using
<code>setFrameRate()</code> doesn’t guarantee your app will achieve the requested frame rate.</p>
<h3>What is the potential impact?</h3>
<ul>
  <li> <em>Usability</em>: the device may run out of battery faster than expected. </li>
  <li> <em>Sustainability</em>: the extra battery usage has a negative impact on the environment. </li>
</ul>
<h2>How to fix it</h2>
<p>Use a frame rate of maximum 60Hz, unless you have a strong reason to used higher rates. Valid exceptions are <em>gaming apps</em>, especially those
with fast-paced action or high-quality graphics, or <em>AR/VR apps</em>.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
public class MainActivity extends AppCompatActivity {

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        SurfaceView surfaceView = findViewById(R.id.my_surface_view);
        Surface surface = surfaceView.getHolder().getSurface();

        surface.setFrameRate(90.0f, Surface.FRAME_RATE_COMPATIBILITY_FIXED_SOURCE); // Noncompliant
    }
}
```
<h4>Compliant solution</h4>

```java
public class MainActivity extends AppCompatActivity {

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        SurfaceView surfaceView = findViewById(R.id.my_surface_view);
        Surface surface = surfaceView.getHolder().getSurface();

        surface.setFrameRate(60.0f, Surface.FRAME_RATE_COMPATIBILITY_FIXED_SOURCE); // Compliant
    }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://developer.android.com/media/optimize/performance/frame-rate">Android for Developers: Frame Rate</a> </li>
  <li> <a
  href="https://developer.apple.com/library/archive/documentation/3DDrawing/Conceptual/MTLBestPracticesGuide/FrameRate.html#//apple_ref/doc/uid/TP40016642-CH23-SW1">Developer Apple - Frame Rate</a> </li>
</ul>