<h2>Why is this an issue?</h2>
<p>Assertions are statements that check whether certain conditions are true. They are used to validate that the actual results of a code snippet match
the expected outcomes. By using assertions, developers can ensure that their code behaves as intended and identify potential bugs or issues early in
the development process.</p>
<p>The convention for passing assertion arguments is to pass the expected value as the first argument and the actual value as the second argument.
This convention is based on the idea that the expected value is what the code is supposed to produce, and the actual value is what the code produces.
By passing the expected value first, it is easier to understand the intent of the assertion and to quickly identify any errors that may be present.
Additionally, many testing frameworks and libraries expect assertion arguments to be passed in this order, so following the convention can help ensure
that your code works correctly with these tools.</p>
<h3>What is the potential impact?</h3>
<p>Having the expected value and the actual value in the wrong order will not alter the outcome of tests, (succeed/fail when it should) but the error
messages will contain misleading information.</p>
<p>This rule raises an issue when the actual argument to an assertions library method is a hard-coded value and the expected argument is not.</p>
<h2>How to fix it</h2>
<p>You should provide the assertion methods with a hard-coded value as the expected value, while the actual value of the assertion should derive from
the portion of code that you want to test.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```js
const assert = require('chai').assert;
const expect = require('chai').expect;
const should = require('chai').should();

it("inverts arguments", function() {
    assert.equal(42, aNumber); // Noncompliant: actual value is passed as first argument and expected as second argument
    expect(42).to.equal(aNumber); // Noncompliant: actual value is passed as first argument and expected as second argument
    should.fail(42, aNumber);  // Noncompliant: actual value is passed as first argument and expected as second argument
});
```
<h4>Compliant solution</h4>
<p>Swap the order of the assertion arguments so that the expected value is passed as the first argument and the actual value is passed as the second
argument.</p>

```js
const assert = require('chai').assert;
const expect = require('chai').expect;
const should = require('chai').should();

it("inverts arguments", function() {
    assert.equal(aNumber, 42);
    expect(aNumber).to.equal(42);
    should.fail(aNumber, 42);
});
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Node.js Documentation - <a href="https://nodejs.org/api/assert.html">Assert</a> </li>
  <li> Chai.js Documentation - <a href="https://www.chaijs.com/api/assert">Assert</a> </li>
  <li> Chai.js Documentation - <a href="https://www.chaijs.com/api/bdd"><code>expect</code> and <code>should</code></a> </li>
</ul>