<h2>Why is this an issue?</h2>
<p>There are two ways to define asynchronous functions in Kotlin:</p>
<ul>
  <li> using the modifier <code>suspend</code> in the function declaration </li>
  <li> creating an extension function on <code>CoroutineScope</code> (or passing it as a parameter) </li>
</ul>
<p>The <code>suspend</code> modifier is generally used for functions that might take some time to complete. The caller coroutine might be potentially
suspended.</p>
<p>Functions that return results immediately but start a coroutine in the background should be written as extension functions on
<code>CoroutineScope</code>. At the same time, these functions should not be declared <code>suspend</code>, as suspending functions should not leave
running background tasks behind.</p>
<h3>Noncompliant code example</h3>

```kotlin
suspend fun CoroutineScope.f(): Int {
    val resource1 = loadResource1()
    val resource2 = loadResource2()
    return resource1.size + resource2.size
}
```
<h3>Compliant solution</h3>
<p>Using <code>suspend</code>:</p>

```kotlin
suspend fun f(): Int {
    val resource1 = loadResource1()
    val resource2 = loadResource2()
    return resource1.size + resource2.size
}
```
<p>Using extension on <code>CoroutineScope</code>:</p>

```kotlin
fun CoroutineScope.f(): Deferred<Int> = async {
    val resource1 = loadResource1()
    val resource2 = loadResource2()
    resource1.size + resource2.size
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://elizarov.medium.com/coroutine-context-and-scope-c8b255d59055">Coroutine Context and Scope</a> </li>
</ul>