<h2>Why is this an issue?</h2>
<p>Kotlin provides the operators <code>as</code> and <code>as?</code> to cast an expression to a specific type, and <code>is</code> to check the
assignment compatibility with a type. These operators are used for downcasts, smart casts, and run-time type checking.</p>
<p>In case the <code>as</code> or <code>as?</code> operator is used for casting between incompatible types, that is, distinct types and neither being
a subtype of the other, the cast will never succeed but always throw a <code>ClassCastException</code>. This results in dead code and is likely a
symptom of wrong program logic.</p>
<p>Likewise, the <code>is</code> operator is redundant and will never return <code>true</code> if the type of the expression on the left side is
incompatible with the type on the right.</p>
<h3>What is the potential impact?</h3>
<h4>Code redundancy</h4>
<p>Since the operation will never succeed, it is pointless to use it. It also leads to dead code branches because <code>as</code> will always break
the regular control flow with an exception, while conditions with <code>is</code> will never or always be satisfied.</p>
<h4>Wrong logic</h4>
<p>Type casts and type checks that can never succeed are likely a symptom of wrong program logic. Developers will not have intended the redundancy of
the type check or type cast, but it might result from an error elsewhere.</p>
<h2>How to fix it</h2>
<p>Remove the operator and all dead code branches that result from it, or investigate why the expression that is type cast or type checked has an
unexpected compile-time type.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```kotlin
fun types(value: String, elements: List<String>) {
    val a: Int = value as Int // Noncompliant, throws ClassCastException
    val b: Int = value as? Int // Noncompliant, will always be null

    val text = if (value is Int) { // Noncomplient, then-branch is dead code
        "impossible"
    } else {
        "happens always"
    }
}
```
<h4>Compliant solution</h4>

```kotlin
fun types(value: Number, elements: List<Number>) {
    val a: Int = value as Int // Compliant, a Number instance could be an Int
    val b: Int = value as? Int // Compliant, a Number instance could be an Int

    val text = if (value is Int) { // Compliant, both branches reachable
        "impossible"
    } else {
        "happens always"
    }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://kotlinlang.org/docs/typecasts.html">Kotlin API Docs, Type checks and casts</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://en.wikipedia.org/wiki/Liskov_substitution_principle">Wikipedia, Liskov substitution principle</a> </li>
</ul>