<p>This rule raises an issue when a generic exception (such as <code>ErrorException</code>, <code>RuntimeException</code> or <code>Exception</code>)
is thrown.</p>
<h2>Why is this an issue?</h2>
<p>Throwing generic exceptions such as <code>Error</code>, <code>RuntimeException</code>, <code>Throwable</code>, and <code>Exception</code> will have
a negative impact on any code trying to catch these exceptions.</p>
<p>From a consumer perspective, it is generally a best practice to only catch exceptions you intend to handle. Other exceptions should ideally be let
to propagate up the stack trace so that they can be dealt with appropriately. When a generic exception is thrown, it forces consumers to catch
exceptions they do not intend to handle, which they then have to re-throw.</p>
<p>Besides, when working with a generic type of exception, the only way to distinguish between multiple exceptions is to check their message, which is
error-prone and difficult to maintain. Legitimate exceptions may be unintentionally silenced and errors may be hidden.</p>
<p>When throwing an exception, it is therefore recommended to throw the most specific exception possible so that it can be handled intentionally by
consumers.</p>
<h2>How to fix it</h2>
<p>To fix this issue, make sure to throw specific exceptions that are relevant to the context in which they arise. It is recommended to either:</p>
<ul>
  <li> Throw a subtype of <code>Exception</code> that already exists in the Standard PHP Library. For instance <code>InvalidArgumentException</code>
  could be raised when an unexpected argument is provided to a function. </li>
  <li> Define a custom exception type that derives from <code>Exception</code> or one of its subclasses. </li>
</ul>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```php
function checkValue($value) {
    if ($value == 42) {
        throw new Exception("Value is 42"); // Noncompliant: This will be difficult for consumers to handle
    }
}
```
<h4>Compliant solution</h4>

```php
function checkValue($value) {
    if ($value == 42) {
        throw new UnexpectedValueException("Value is 42"); // Compliant
    }
}
```
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/397">CWE-397 Declaration of Throws for Generic Exception</a> </li>
</ul>