<p>Secret leaks often occur when a sensitive piece of authentication data is stored with the source code of an application. Considering the source
code is intended to be deployed across multiple assets, including source code repositories or application hosting servers, the secrets might get
exposed to an unintended audience.</p>
<h2>Why is this an issue?</h2>
<p>In most cases, trust boundaries are violated when a secret is exposed in a source code repository or an uncontrolled deployment environment.
Unintended people who don’t need to know the secret might get access to it. They might then be able to use it to gain unwanted access to associated
services or resources.</p>
<p>The trust issue can be more or less severe depending on the people’s role and entitlement.</p>
<h3>What is the potential impact?</h3>
<p>The consequences vary greatly depending on the situation and the secret-exposed audience. Still, two main scenarios should be considered.</p>
<h4>Financial loss</h4>
<p>Financial losses can occur when a secret is used to access a paid third-party-provided service and is disclosed as part of the source code of
client applications. Having the secret, each user of the application will be able to use it without limit to use the third party service to their own
need, including in a way that was not expected.</p>
<p>This additional use of the secret will lead to added costs with the service provider.</p>
<p>Moreover, when rate or volume limiting is set up on the provider side, this additional use can prevent the regular operation of the affected
application. This might result in a partial denial of service for all the application’s users.</p>
<h4>Application’s security downgrade</h4>
<p>A downgrade can happen when the disclosed secret is used to protect security-sensitive assets or features of the application. Depending on the
affected asset or feature, the practical impact can range from a sensitive information leak to a complete takeover of the application, its hosting
server or another linked component.</p>
<p>For example, an application that would disclose a secret used to sign user authentication tokens would be at risk of user identity impersonation.
An attacker accessing the leaked secret could sign session tokens for arbitrary users and take over their privileges and entitlements.</p>
<h2>How to fix it</h2>
<p><strong>Revoke the secret</strong></p>
<p>Revoke any leaked secrets and remove them from the application source code.</p>
<p>Before revoking the secret, ensure that no other applications or processes are using it. Other usages of the secret will also be impacted when the
secret is revoked.</p>
<p><strong>Analyze recent secret use</strong></p>
<p>When available, analyze authentication logs to identify any unintended or malicious use of the secret since its disclosure date. Doing this will
allow determining if an attacker took advantage of the leaked secret and to what extent.</p>
<p>This operation should be part of a global incident response process.</p>
<p><strong>Use a secret vault</strong></p>
<p>A secret vault should be used to generate and store the new secret. This will ensure the secret’s security and prevent any further unexpected
disclosure.</p>
<p>Depending on the development platform and the leaked secret type, multiple solutions are currently available.</p>
<h3>Code examples</h3>
<p>The following code example is noncompliant because it uses a hardcoded secret value.</p>
<h4>Noncompliant code example</h4>

```php
use Defuse\Crypto\KeyOrPassword;

function createKey() {
    $password = "3xAmpl3";  // Noncompliant
    return KeyOrPassword::createFromPassword($password);
}
```
<h4>Compliant solution</h4>

```php
use Defuse\Crypto\KeyOrPassword;

function createKey() {
    $password = $_ENV["SECRET"]
    return KeyOrPassword::createFromPassword($password);
}
```
<h3>How does this work?</h3>
<p>While the noncompliant code example contains a hard-coded password, the compliant solution retrieves the secret’s value from its environment. This
allows to have an environment-dependent secret value and avoids storing the password in the source code itself.</p>
<p>Depending on the application and its underlying infrastructure, how the secret gets added to the environment might change.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> AWS Documentation - <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">What is AWS Secrets Manager</a> </li>
  <li> Azure Documentation - <a href="https://learn.microsoft.com/en-us/azure/key-vault/">Azure Key Vault</a> </li>
  <li> Google Cloud - <a href="https://cloud.google.com/secret-manager/docs">Secret Manager documentation</a> </li>
  <li> HashiCorp Developer - <a href="https://developer.hashicorp.com/vault/docs">Vault Documentation</a> </li>
  <li> Symfony - <a href="https://symfony.com/doc/current/configuration/secrets.html">How to Keep Sensitive Information Secret</a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/">Top 10 2021 - Category A7 - Identification and
  Authentication Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A2_2017-Broken_Authentication">Top 10 2017 - Category A2 - Broken
  Authentication</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/798">CWE-798 - Use of Hard-coded Credentials</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/259">CWE-259 - Use of Hard-coded Password</a> </li>
</ul>