<h2>Why is this an issue?</h2>
<p>In TypeScript, <code>any</code> is a type that is used when the type of a variable is unknown or could be of any type. It allows you to opt-out of
type-checking and let the values pass through compile-time checks. In other words, it prevents the compiler from reporting type errors, which can lead
to runtime errors.</p>
<p>On the other hand, <code>unknown</code> is a type-safe alternative to <code>any</code>. It forces you to perform certain checks before performing
operations on variables of type <code>unknown</code>. This means you can’t accidentally perform arbitrary operations on variables of type
<code>unknown</code>, which helps prevent runtime errors.</p>
<p>It’s generally recommended to avoid using <code>any</code> when possible, and instead use more specific types or generics for better type safety.
If you want to maintain type safety, it’s better to use <code>unknown</code> instead of <code>any</code>.</p>

```ts
function logValue(value: any) { // Noncompliant: 'value' is not type-checked
  console.log(value);
}

logValue(123);
logValue('Hello');
```
<p>You should use <code>unknown</code> instead of <code>any</code> and narrow it down with type guards.</p>

```ts
function logValue(value: unknown) {
  if (typeof value === 'number') {
    console.log(value.toFixed(2));
  } else if (typeof value === 'string') {
    console.log(value.trim());
  }
}

logValue(123);
logValue('Hello');
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> TypeScript Documentation - <a href="https://www.typescriptlang.org/docs/handbook/2/functions.html#unknown"><code>unknown</code></a> </li>
  <li> TypeScript Documentation - <a href="https://www.typescriptlang.org/docs/handbook/2/everyday-types.html#any"><code>any</code></a> </li>
  <li> TypeScript Documentation - <a href="https://www.typescriptlang.org/docs/handbook/2/narrowing.html">Narrowing</a> </li>
</ul>