"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RULE_NAME = exports.MESSAGE_ID = void 0;
const utils_1 = require("@angular-eslint/utils");
const create_eslint_rule_1 = require("../utils/create-eslint-rule");
const get_dom_elements_1 = require("../utils/get-dom-elements");
exports.MESSAGE_ID = 'preferSelfClosingTags';
exports.RULE_NAME = 'prefer-self-closing-tags';
exports.default = (0, create_eslint_rule_1.createESLintRule)({
    name: exports.RULE_NAME,
    meta: {
        type: 'layout',
        docs: {
            description: 'Ensures that self-closing tags are used for elements with a closing tag but no content.',
        },
        fixable: 'code',
        schema: [],
        messages: {
            [exports.MESSAGE_ID]: 'Use self-closing tags for elements with a closing tag but no content.',
        },
    },
    defaultOptions: [],
    create(context) {
        const parserServices = (0, utils_1.getTemplateParserServices)(context);
        return {
            'Element$1, Template, Content'(node) {
                if (isContentNode(node)) {
                    processContentNode(node);
                }
                else {
                    // Ignore native elements.
                    if ('name' in node && (0, get_dom_elements_1.getDomElements)().has(node.name)) {
                        return;
                    }
                    processElementOrTemplateNode(node);
                }
            },
        };
        function processElementOrTemplateNode(node) {
            const { children, startSourceSpan, endSourceSpan } = node;
            const noContent = !children.length ||
                children.every((node) => {
                    const text = node.value;
                    // If the node has no value, or only whitespace,
                    // we can consider it empty.
                    return (typeof text === 'string' && text.replace(/\n/g, '').trim() === '');
                });
            const noCloseTag = !endSourceSpan ||
                (startSourceSpan.start.offset === endSourceSpan.start.offset &&
                    startSourceSpan.end.offset === endSourceSpan.end.offset);
            if (!noContent || noCloseTag) {
                return;
            }
            // HTML tags always have more than two characters
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            const openingTagLastChar = startSourceSpan.toString().at(-2);
            const closingTagPrefix = getClosingTagPrefix(openingTagLastChar);
            context.report({
                loc: parserServices.convertNodeSourceSpanToLoc(endSourceSpan),
                messageId: exports.MESSAGE_ID,
                fix: (fixer) => fixer.replaceTextRange([startSourceSpan.end.offset - 1, endSourceSpan.end.offset], closingTagPrefix + '/>'),
            });
        }
        function processContentNode(node) {
            var _a, _b;
            const { sourceSpan } = node;
            const ngContentCloseTag = '</ng-content>';
            if (sourceSpan.toString().includes(ngContentCloseTag)) {
                // content nodes can only contain whitespaces
                const content = (_b = (_a = sourceSpan
                    .toString()
                    .match(/>(\s*)</m)) === null || _a === void 0 ? void 0 : _a.at(1)) !== null && _b !== void 0 ? _b : '';
                const openingTagLastChar = 
                // This is more than the minimum length of a ng-content element
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                sourceSpan
                    .toString()
                    .at(-2 - ngContentCloseTag.length - content.length);
                const closingTagPrefix = getClosingTagPrefix(openingTagLastChar);
                context.report({
                    // content nodes don't have information about startSourceSpan and endSourceSpan,
                    // so we need to calculate it by our own
                    loc: {
                        start: {
                            line: sourceSpan.end.line + 1,
                            column: sourceSpan.end.col - ngContentCloseTag.length,
                        },
                        end: {
                            line: sourceSpan.end.line + 1,
                            column: sourceSpan.end.col,
                        },
                    },
                    messageId: exports.MESSAGE_ID,
                    fix: (fixer) => fixer.replaceTextRange([
                        sourceSpan.end.offset -
                            ngContentCloseTag.length -
                            content.length -
                            1,
                        sourceSpan.end.offset,
                    ], closingTagPrefix + '/>'),
                });
            }
        }
    },
});
function isContentNode(node) {
    return 'name' in node && node.name === 'ng-content';
}
function getClosingTagPrefix(openingTagLastChar) {
    const hasOwnWhitespace = openingTagLastChar.trim() === '';
    return hasOwnWhitespace ? '' : ' ';
}
