/*
 * Decompiled with CFR 0.152.
 */
package com.teamscale.index.s3;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import java.net.URI;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.conqat.lib.commons.collections.Pair;
import org.conqat.lib.commons.string.StringUtils;
import org.jetbrains.annotations.VisibleForTesting;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

public class S3UriParser {
    public static final String S3_SCHEME = "s3";
    public static final String GS_SCHEME = "gs";
    @VisibleForTesting
    static final String GOOGLE_CLOUD_STORAGE_ENDPOINT = "https://storage.googleapis.com";
    private static final Pattern BUCKET_STYLE_GCS_ENDPOINT_PATTERN = Pattern.compile("https://([^.]+)\\.storage\\.googleapis\\.com");

    public static ParsedUri parseUri(URI uri) {
        String scheme = S3UriParser.mapScheme(uri);
        if (scheme.equalsIgnoreCase(S3_SCHEME)) {
            return S3UriParser.parseGlobalS3Uri(uri);
        }
        if (scheme.equalsIgnoreCase(GS_SCHEME)) {
            return S3UriParser.parseGsUri(uri);
        }
        if (uri.getHost().matches("[^.]+\\.s3[.-].*")) {
            return S3UriParser.parseVirtualHostStyleS3HttpUri(uri);
        }
        Matcher gcsBucketStyleUriMatcher = BUCKET_STYLE_GCS_ENDPOINT_PATTERN.matcher(S3UriParser.parseEndpoint(uri, true));
        if (gcsBucketStyleUriMatcher.matches()) {
            return S3UriParser.parseGcsBucketStyleUri(uri, gcsBucketStyleUriMatcher);
        }
        return S3UriParser.parsePathStyleS3HttpUri(uri);
    }

    private static String mapScheme(URI uri) {
        Preconditions.checkArgument((uri.getScheme() != null ? 1 : 0) != 0, (Object)"URI scheme should not be null.");
        return switch (uri.getScheme()) {
            case "s3+http", "http" -> "http";
            case "s3+https", "https" -> "https";
            case S3_SCHEME -> S3_SCHEME;
            case GS_SCHEME -> GS_SCHEME;
            default -> throw new IllegalArgumentException("Unsupported URI scheme: " + uri.getScheme());
        };
    }

    private static ParsedUri parseGlobalS3Uri(URI uri) {
        String bucket = uri.getAuthority();
        String key = "";
        String path = uri.getPath();
        if (!StringUtils.isEmpty((String)path)) {
            key = S3UriParser.dropLeadingSlash(path);
        }
        return new ParsedUri(S3UriParser.parseEndpoint(uri, true), null, null, bucket, key);
    }

    private static ParsedUri parseGsUri(URI uri) {
        return new ParsedUri(GOOGLE_CLOUD_STORAGE_ENDPOINT, null, null, uri.getAuthority(), S3UriParser.dropLeadingSlash(uri.getPath()));
    }

    private static ParsedUri parseVirtualHostStyleS3HttpUri(URI uri) {
        String endpoint = S3UriParser.parseEndpoint(uri, false);
        AuthKeys authKeys = S3UriParser.parseAuthKeys(uri);
        String host = uri.getHost();
        String bucket = StringUtils.getFirstPart((String)host, (char)'.');
        String key = StringUtils.stripPrefix((String)uri.getPath(), (String)"/");
        return new ParsedUri(endpoint, authKeys.accessKey(), authKeys.secretKey(), bucket, key);
    }

    private static ParsedUri parseGcsBucketStyleUri(URI uri, Matcher gcsBucketStyleUriMatcher) {
        return new ParsedUri(GOOGLE_CLOUD_STORAGE_ENDPOINT, null, null, gcsBucketStyleUriMatcher.group(1), S3UriParser.dropLeadingSlash(uri.getPath()));
    }

    private static ParsedUri parsePathStyleS3HttpUri(URI uri) {
        String endpoint = S3UriParser.parseEndpoint(uri, true);
        AuthKeys authKeys = S3UriParser.parseAuthKeys(uri);
        BucketAndKey bucketAndKey = S3UriParser.extractBucketAndKeyFromPath(uri);
        return new ParsedUri(endpoint, authKeys.accessKey(), authKeys.secretKey(), bucketAndKey.bucket(), bucketAndKey.key());
    }

    private static AuthKeys parseAuthKeys(URI uri) {
        String userInfo = uri.getUserInfo();
        if (userInfo == null) {
            return new AuthKeys(null, null);
        }
        Pair accessKeyAndSecretKey = StringUtils.splitAtFirst((String)userInfo, (char)':');
        return new AuthKeys((String)accessKeyAndSecretKey.getFirst(), (String)accessKeyAndSecretKey.getSecond());
    }

    private static BucketAndKey extractBucketAndKeyFromPath(URI uri) {
        String path = uri.getPath();
        if (StringUtils.isEmpty((String)path)) {
            return new BucketAndKey("", "");
        }
        path = S3UriParser.dropLeadingSlash(path);
        Pair bucketAndKey = StringUtils.splitAtFirst((String)path, (char)'/');
        return new BucketAndKey((String)bucketAndKey.getFirst(), (String)bucketAndKey.getSecond());
    }

    private static @NonNull String dropLeadingSlash(String path) {
        return StringUtils.stripPrefix((String)StringUtils.emptyIfNull((String)path), (String)"/");
    }

    private static String parseEndpoint(URI uri, boolean pathStyle) {
        String scheme = S3UriParser.mapScheme(uri);
        StringBuilder endpoint = new StringBuilder();
        endpoint.append(scheme).append("://");
        if (scheme.equals(S3_SCHEME)) {
            return endpoint.toString();
        }
        String host = uri.getHost();
        if (pathStyle) {
            endpoint.append(host);
        } else {
            int indexOfDot = host.indexOf(46);
            endpoint.append(host.substring(indexOfDot + 1));
        }
        if (uri.getPort() >= 0) {
            endpoint.append(':').append(uri.getPort());
        }
        return endpoint.toString();
    }

    public static class ParsedUri {
        private final @NonNull String endpoint;
        private final @Nullable String accessKey;
        private final @Nullable String secretKey;
        private final @NonNull String bucket;
        private final @NonNull String key;

        public ParsedUri(@NonNull String endpoint, @Nullable String accessKey, @Nullable String secretKey, @NonNull String bucket, @NonNull String key) {
            this.endpoint = Objects.requireNonNull(endpoint);
            this.accessKey = accessKey;
            this.secretKey = secretKey;
            this.bucket = Objects.requireNonNull(bucket);
            this.key = Objects.requireNonNull(key);
        }

        public ParsedUri(String endpoint, String bucket, String key) {
            this(endpoint, null, null, bucket, key);
        }

        public @NonNull String getEndpoint() {
            return this.endpoint;
        }

        public @Nullable String getAccessKey() {
            return this.accessKey;
        }

        public @Nullable String getSecretKey() {
            return this.secretKey;
        }

        public @NonNull String getBucket() {
            return this.bucket;
        }

        public @NonNull String getKey() {
            return this.key;
        }

        public final int hashCode() {
            return Objects.hash(this.endpoint, this.accessKey, this.secretKey, this.bucket, this.key);
        }

        public final boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (other == null) {
                return false;
            }
            if (!(other instanceof ParsedUri)) {
                return false;
            }
            ParsedUri that = (ParsedUri)other;
            return Objects.equals(this.endpoint, that.endpoint) && Objects.equals(this.accessKey, that.accessKey) && Objects.equals(this.secretKey, that.secretKey) && Objects.equals(this.bucket, that.bucket) && Objects.equals(this.key, that.key);
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("endpoint", (Object)this.endpoint).add("accessKey", (Object)this.accessKey).add("secretKey", (Object)this.secretKey).add("bucket", (Object)this.bucket).add("key", (Object)this.key).omitNullValues().toString();
        }

        public boolean isGlobalS3Uri() {
            return this.endpoint.equals("s3://");
        }

        public boolean isGoogleCloudStorageUri() {
            return S3UriParser.GOOGLE_CLOUD_STORAGE_ENDPOINT.equals(this.endpoint);
        }

        public String toLookupKey() {
            if (this.isGlobalS3Uri()) {
                return "https://" + this.bucket + ".s3.amazonaws.com/";
            }
            String virtualHostUrl = StringUtils.ensureStartsWith((String)URI.create(this.endpoint).getHost(), (String)"s3.");
            return StringUtils.ensureEndsWith((String)("https://" + this.bucket + "." + virtualHostUrl), (String)"/");
        }
    }

    private record AuthKeys(@Nullable String accessKey, @Nullable String secretKey) {
    }

    private record BucketAndKey(String bucket, String key) {
    }
}

