Disallow certain types in boolean expressions.


Forbids usage of non-boolean types in expressions where a boolean is expected.
`boolean` and `never` types are always allowed.
Additional types which are considered safe in a boolean context can be configured via options.

The following nodes are considered boolean expressions and their type is checked:

- Argument to the logical negation operator (`!arg`).
- The condition in a conditional expression (`cond ? x : y`).
- Conditions for `if`, `for`, `while`, and `do-while` statements.
- Operands of logical binary operators (`lhs || rhs` and `lhs && rhs`).
  - Right-hand side operand is ignored when it's not a descendant of another boolean expression.
    This is to allow usage of boolean operators for their short-circuiting behavior.

## Examples

<!--tabs-->

### ❌ Incorrect

```ts
// nullable numbers are considered unsafe by default
let num: number | undefined = 0;
if (num) {
  console.log('num is defined');
}

// nullable strings are considered unsafe by default
let str: string | null = null;
if (!str) {
  console.log('str is empty');
}

// nullable booleans are considered unsafe by default
function foo(bool?: boolean) {
  if (bool) {
    bar();
  }
}

// `any`, unconstrained generics and unions of more than one primitive type are disallowed
const foo = <T>(arg: T) => (arg ? 1 : 0);

// always-truthy and always-falsy types are disallowed
let obj = {};
while (obj) {
  obj = getObj();
}
```

### ✅ Correct

```tsx
// Using logical operator short-circuiting is allowed
const Component = () => {
  const entry = map.get('foo') || {};
  return entry && <p>Name: {entry.name}</p>;
};

// nullable values should be checked explicitly against null or undefined
let num: number | undefined = 0;
if (num != null) {
  console.log('num is defined');
}

let str: string | null = null;
if (str != null && !str) {
  console.log('str is empty');
}

function foo(bool?: boolean) {
  if (bool ?? false) {
    bar();
  }
}

// `any` types should be cast to boolean explicitly
const foo = (arg: any) => (Boolean(arg) ? 1 : 0);
```

## Fixes and Suggestions

This rule provides following fixes and suggestions for particular types in boolean context:

- `boolean` - Always allowed - no fix needed.
- `string` - (when `allowString` is `false`) - Provides following suggestions:
  - Change condition to check string's length (`str` → `str.length > 0`)
  - Change condition to check for empty string (`str` → `str !== ""`)
  - Explicitly cast value to a boolean (`str` → `Boolean(str)`)
- `number` - (when `allowNumber` is `false`):
  - For `array.length` - Provides **autofix**:
    - Change condition to check for 0 (`array.length` → `array.length > 0`)
  - For other number values - Provides following suggestions:
    - Change condition to check for 0 (`num` → `num !== 0`)
    - Change condition to check for NaN (`num` → `!Number.isNaN(num)`)
    - Explicitly cast value to a boolean (`num` → `Boolean(num)`)
- `object | null | undefined` - (when `allowNullableObject` is `false`) - Provides **autofix**:
  - Change condition to check for null/undefined (`maybeObj` → `maybeObj != null`)
- `boolean | null | undefined` - Provides following suggestions:
  - Explicitly treat nullish value the same as false (`maybeBool` → `maybeBool ?? false`)
  - Change condition to check for true/false (`maybeBool` → `maybeBool === true`)
- `string | null | undefined` - Provides following suggestions:
  - Change condition to check for null/undefined (`maybeStr` → `maybeStr != null`)
  - Explicitly treat nullish value the same as an empty string (`maybeStr` → `maybeStr ?? ""`)
  - Explicitly cast value to a boolean (`maybeStr` → `Boolean(maybeStr)`)
- `number | null | undefined` - Provides following suggestions:
  - Change condition to check for null/undefined (`maybeNum` → `maybeNum != null`)
  - Explicitly treat nullish value the same as 0 (`maybeNum` → `maybeNum ?? 0`)
  - Explicitly cast value to a boolean (`maybeNum` → `Boolean(maybeNum)`)
- `any` and `unknown` - Provides following suggestions:
  - Explicitly cast value to a boolean (`value` → `Boolean(value)`)

## Related To

- [no-unnecessary-condition](https://typescript-eslint.io/rules/no-unnecessary-condition) - Similar rule which reports always-truthy and always-falsy values in conditions
