<h2>Why is this an issue?</h2>
<p>When <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/selection-statements#the-switch-statement">switch</a>
statements have large sets of multi-line <code>case</code> clauses, the code becomes hard to read and maintain.</p>
<p>For example, the <a href="https://www.sonarsource.com/docs/CognitiveComplexity.pdf">Cognitive Complexity</a> is going to be particularly high.</p>
<p>In such scenarios, it’s better to refactor the <code>switch</code> to only have single-line case clauses.</p>
<p>When all the <code>case</code> clauses of a <code>switch</code> statement are single-line, the readability of the code is not affected. Moreover,
<code>switch</code> statements with single-line <code>case</code> clauses can easily be converted into <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/switch-expression"><code>switch</code> expressions</a>, which are
more concise for assignment and avoid the need for <code>break</code> statements.</p>
<h3>Exceptions</h3>
<p>This rule ignores:</p>
<ul>
  <li> <code>switch</code> statements over <code>Enum</code> arguments </li>
  <li> fall-through cases </li>
  <li> <code>return</code>, <code>break</code> and <code>throw</code> statements in case clauses </li>
</ul>
<h2>How to fix it</h2>
<p>Extract the logic of multi-line <code>case</code> clauses into separate methods.</p>
<h3>Code examples</h3>
<p>The examples below use the "Maximum number of case" property set to <code>4</code>.</p>
<p>Note that from C# 8, you can use <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/switch-expression"><code>switch</code> expression</a>.</p>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
public int MapChar(char ch, int value)
{
    switch(ch) // Noncompliant
    {
        case 'a':
            return 1;
        case 'b':
            return 2;
        case 'c':
            return 3;
        // ...
        case '-':
            if (value &gt; 10)
            {
                return 42;
            }
            else if (value &lt; 5 &amp;&amp; value &gt; 1)
            {
                return 21;
            }
            return 99;
        default:
            return 1000;
    }
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
public int MapChar(char ch, int value)
{
    switch(ch) // Compliant: All 5 cases are single line statements
    {
        case 'a':
            return 1;
        case 'b':
            return 2;
        case 'c':
            return 3;
        // ...
        case '-':
            return HandleDash(value);
        default:
            return 1000;
    }
}

private int HandleDash(int value)
{
    if (value &gt; 10)
    {
        return 42;
    }
    else if (value &lt; 5 &amp;&amp; value &gt; 1)
    {
        return 21;
    }
    return 99;
}
</pre>
<p>For this example, a <code>switch</code> expression is more concise and clear:</p>
<pre>
public int MapChar(char ch, int value) =&gt;
    ch switch // Compliant
    {
        'a' =&gt; 1,
        'b' =&gt; 2,
        'c' =&gt; 3,
        // ...
        '-' =&gt; HandleDash(value),
        _ =&gt; 1000,
    };

private int HandleDash(int value)
{
    if (value &gt; 10)
    {
        return 42;
    }
    else if (value &lt; 5 &amp;&amp; value &gt; 1)
    {
        return 21;
    }
    return 99;
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Sonar - <a href="https://www.sonarsource.com/docs/CognitiveComplexity.pdf">Cognitive Complexity</a> </li>
  <li> Microsoft Learn - <a
  href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/selection-statements#the-switch-statement">The
  <code>switch</code> statement</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/switch-expression">C#: Switch
  Expression</a> </li>
</ul>

