<p>The use of a non-standard algorithm is dangerous because a determined attacker may be able to break the algorithm and compromise whatever data has
been protected. Standard algorithms like <code>AES</code>, <code>RSA</code>, <code>SHA</code>, …​ should be used instead.</p>
<p>This rule tracks custom implementation of these types from <code>System.Security.Cryptography</code> namespace:</p>
<ul>
  <li> <code>AsymmetricAlgorithm</code> </li>
  <li> <code>AsymmetricKeyExchangeDeformatter</code> </li>
  <li> <code>AsymmetricKeyExchangeFormatter</code> </li>
  <li> <code>AsymmetricSignatureDeformatter</code> </li>
  <li> <code>AsymmetricSignatureFormatter</code> </li>
  <li> <code>DeriveBytes</code> </li>
  <li> <code>HashAlgorithm</code> </li>
  <li> <code>ICryptoTransform</code> </li>
  <li> <code>SymmetricAlgorithm</code> </li>
</ul>
<h2>Recommended Secure Coding Practices</h2>
<ul>
  <li> Use a standard algorithm instead of creating a custom one. </li>
</ul>
<h2>Sensitive Code Example</h2>
<pre>
public class CustomHash : HashAlgorithm // Noncompliant
{
    private byte[] result;

    public override void Initialize() =&gt; result = null;
    protected override byte[] HashFinal() =&gt; result;

    protected override void HashCore(byte[] array, int ibStart, int cbSize) =&gt;
        result ??= array.Take(8).ToArray();
}
</pre>
<h2>Compliant Solution</h2>
<pre>
SHA256 mySHA256 = SHA256.Create()
</pre>
<h2>See</h2>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">Top 10 2021 Category A2 - Cryptographic Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/327">CWE-327 - Use of a Broken or Risky Cryptographic Algorithm</a> </li>
  <li> Derived from FindSecBugs rule <a href="https://h3xstream.github.io/find-sec-bugs/bugs.htm#CUSTOM_MESSAGE_DIGEST">MessageDigest is Custom</a>
  </li>
</ul>

