<p>In Unix, "others" class refers to all users except the owner of the file and the members of the group assigned to this file.</p>
<p>In Windows, "Everyone" group is similar and includes all members of the Authenticated Users group as well as the built-in Guest account, and
several other built-in security accounts.</p>
<p>Granting permissions to these groups can lead to unintended access to files.</p>
<h2>Ask Yourself Whether</h2>
<ul>
  <li> The application is designed to be run on a multi-user environment. </li>
  <li> Corresponding files and directories may contain confidential information. </li>
</ul>
<p>There is a risk if you answered yes to any of those questions.</p>
<h2>Recommended Secure Coding Practices</h2>
<p>The most restrictive possible permissions should be assigned to files and directories.</p>
<h2>Sensitive Code Example</h2>
<p>.Net Framework:</p>
<pre>
var unsafeAccessRule = new FileSystemAccessRule("Everyone", FileSystemRights.FullControl, AccessControlType.Allow);

var fileSecurity = File.GetAccessControl("path");
fileSecurity.AddAccessRule(unsafeAccessRule); // Sensitive
fileSecurity.SetAccessRule(unsafeAccessRule); // Sensitive
File.SetAccessControl("fileName", fileSecurity);
</pre>
<p>.Net / .Net Core</p>
<pre>
var fileInfo = new FileInfo("path");
var fileSecurity = fileInfo.GetAccessControl();

fileSecurity.AddAccessRule(new FileSystemAccessRule("Everyone", FileSystemRights.Write, AccessControlType.Allow)); // Sensitive
fileInfo.SetAccessControl(fileSecurity);
</pre>
<p>.Net / .Net Core using Mono.Posix.NETStandard</p>
<pre>
var fileSystemEntry = UnixFileSystemInfo.GetFileSystemEntry("path");
fileSystemEntry.FileAccessPermissions = FileAccessPermissions.OtherReadWriteExecute; // Sensitive
</pre>
<h2>Compliant Solution</h2>
<p>.Net Framework</p>
<pre>
var safeAccessRule = new FileSystemAccessRule("Everyone", FileSystemRights.FullControl, AccessControlType.Deny);

var fileSecurity = File.GetAccessControl("path");
fileSecurity.AddAccessRule(safeAccessRule);
File.SetAccessControl("path", fileSecurity);
</pre>
<p>.Net / .Net Core</p>
<pre>
var safeAccessRule = new FileSystemAccessRule("Everyone", FileSystemRights.FullControl, AccessControlType.Deny);

var fileInfo = new FileInfo("path");
var fileSecurity = fileInfo.GetAccessControl();
fileSecurity.SetAccessRule(safeAccessRule);
fileInfo.SetAccessControl(fileSecurity);
</pre>
<p>.Net / .Net Core using Mono.Posix.NETStandard</p>
<pre>
var fs = UnixFileSystemInfo.GetFileSystemEntry("path");
fs.FileAccessPermissions = FileAccessPermissions.UserExecute;
</pre>
<h2>See</h2>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A01_2021-Broken_Access_Control/">Top 10 2021 Category A1 - Broken Access Control</a> </li>
  <li> OWASP - <a href="https://owasp.org/Top10/A04_2021-Insecure_Design/">Top 10 2021 Category A4 - Insecure Design</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A5_2017-Broken_Access_Control">Top 10 2017 Category A5 - Broken Access Control</a>
  </li>
  <li> <a
  href="https://owasp.org/www-project-web-security-testing-guide/latest/4-Web_Application_Security_Testing/02-Configuration_and_Deployment_Management_Testing/09-Test_File_Permission">OWASP File Permission</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/732">CWE-732 - Incorrect Permission Assignment for Critical Resource</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/266">CWE-266 - Incorrect Privilege Assignment</a> </li>
  <li> STIG Viewer - <a href="https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222430">Application Security and
  Development: V-222430</a> - The application must execute without excessive account permissions. </li>
</ul>

