﻿## SA1624

<table>
<tr>
  <td>TypeName</td>
  <td>SA1624PropertySummaryDocumentationMustOmitSetAccessorWithRestrictedAccess</td>
</tr>
<tr>
  <td>CheckId</td>
  <td>SA1624</td>
</tr>
<tr>
  <td>Category</td>
  <td>Documentation Rules</td>
</tr>
</table>

## Cause

The documentation text within a C# property's `<summary>` tag takes into account all of the accessors within the property, but one of the accessors has limited access.

## Rule description

C# syntax provides a mechanism for inserting documentation for classes and elements directly into the code, through the use of Xml documentation headers. For an introduction to these headers and a description of the header syntax, see the following article: [https://docs.microsoft.com/en-us/dotnet/csharp/programming-guide/xmldoc/xml-documentation-comments](https://docs.microsoft.com/en-us/dotnet/csharp/programming-guide/xmldoc/xml-documentation-comments).

A violation of this rule occurs when one of the accessors within the property has limited access (usually the set accessor), but the summary documentation text for the property still refers to both accessors.

Normally, a property's summary text should begin with wording describing the types of accessors exposed within the property. If the property contains only a get accessor, the summary should begin with the word "Gets". If the property contains only a set accessor, the summary should begin with the word "Sets". If the property exposes both a get and set accessor, the summary text should begin with "Gets or sets".

However, when an accessor within the property is given an access level which is more limited than the access level of the property, this accessor should be omitted from the summary documentation. 

It can sometimes be non-obvious whether the set accessor within a property is actually less accessible than the get accessor. For example, consider the case where a public property is contained within an internal class, and the set accessor is given internal accessor. In effect, both the get and set accessors have the same access level. In this case, the summary documentation should refer to both the get and set accessors, since they effectively have the same access level.

In some situations, the set accessor for a property can have more restricted access than the get accessor. For example:

```csharp
/// <summary>
/// Gets the name of the customer. 
/// </summary>
public string Name
{
    get { return this.name; }
    private set { this.name = value; }
}
```

In this example, the set accessor has been given private access, meaning that it can only be accessed by local members of the class in which it is contained. The get accessor, however, inherits its access from the parent property, thus it can be accessed by any caller, since the property has public access.

In this case, the documentation summary text should avoid referring to the set accessor, since it is not visible to external callers.

StyleCop applies a series of rules to determine when the set accessor should be referenced in the property's summary documentation. In general, these rules require the set accessor to be referenced whenever it is visible to the same set of callers as the get accessor, or whenever it is visible to external classes or inheriting classes. 

The specific rules for determining whether to include the set accessor in the property's summary documentation are:

1. The set accessor has the same access level as the get accessor. For example:

    ```csharp
    /// <summary>
    /// Gets or sets the name of the customer. 
    /// </summary>
    protected string Name
    {
        get { return this.name; }
        set { this.name = value; }
    }
    ```

2. The property is only internally accessible within the assembly, and the set accessor also has internal access. For example:

    ```csharp
    internal class Class1
    {
        /// <summary>
        /// Gets or sets the name of the customer. 
        /// </summary>
        protected string Name
        {
            get { return this.name; }
            internal set { this.name = value; }
        }
    }

    internal class Class1
    {
        public class Class2
        {
            /// <summary>
            /// Gets or sets the name of the customer. 
            /// </summary>
            public string Name
            {
                get { return this.name; }
                internal set { this.name = value; }
            }
        }
    }
    ```

3. The property is private or is contained beneath a private class, and the set accessor has any access modifier other than private. In the example below, the access modifier declared on the set accessor has no meaning, since the set accessor is contained within a private class and thus cannot be seen by other classes outside of Class1. This effectively gives the set accessor the same access level as the get accessor.

    ```csharp
    public class Class1
    {
        private class Class2
        {
            public class Class3
            {
                /// <summary>
                /// Gets or sets the name of the customer. 
                /// </summary>
                public string Name
                {
                    get { return this.name; }
                    internal set { this.name = value; }
                }
            }
        }
    }
    ```

4. Whenever the set accessor has protected or protected internal access, it should be referenced in the documentation. A protected or protected internal set accessor can always been seen by a class inheriting from the class containing the property.

    ```csharp
    internal class Class1
    {
        public class Class2
        {
            /// <summary>
            /// Gets or sets the name of the customer. 
            /// </summary>
            internal string Name
            {
                get { return this.name; }
                protected set { this.name = value; }
            }
        }

        private class Class3 : Class2
        {
            public Class3(string name) { this.Name = name; }
        }
    }
    ```

## How to fix violations

To fix a violation of this rule, update the property's summary text and remove wording which refers to the limited access accessor.

## How to suppress violations

```csharp
[SuppressMessage("StyleCop.CSharp.DocumentationRules", "SA1624:PropertySummaryDocumentationMustOmitSetAccessorWithRestrictedAccess", Justification = "Reviewed.")]
```

```csharp
#pragma warning disable SA1624 // PropertySummaryDocumentationMustOmitSetAccessorWithRestrictedAccess
#pragma warning restore SA1624 // PropertySummaryDocumentationMustOmitSetAccessorWithRestrictedAccess
```
