<h2>Why is this an issue?</h2>
<p>Hard-coding a URI makes it difficult to test a program for a variety of reasons:</p>
<ul>
  <li> path literals are not always portable across operating systems </li>
  <li> a given absolute path may not exist in a specific test environment </li>
  <li> a specified Internet URL may not be available when executing the tests </li>
  <li> production environment filesystems usually differ from the development environment </li>
</ul>
<p>In addition, hard-coded URIs can contain sensitive information, like IP addresses, and they should not be stored in the code.</p>
<p>For all those reasons, a URI should never be hard coded. Instead, it should be replaced by a customizable parameter.</p>
<p>Further, even if the elements of a URI are obtained dynamically, portability can still be limited if the path delimiters are hard-coded.</p>
<p>This rule raises an issue when URIs or path delimiters are hard-coded.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
public class Foo {
  public Collection<User> listUsers() {
    File userList = new File("/home/mylogin/Dev/users.txt"); // Noncompliant
    Collection<User> users = parse(userList);
    return users;
  }
}
```
<h4>Compliant solution</h4>

```java
public class Foo {
  // Configuration is a class that returns customizable properties: it can be mocked to be injected during tests.
  private Configuration config;
  public Foo(Configuration myConfig) {
    this.config = myConfig;
  }
  public Collection<User> listUsers() {
    // Find here the way to get the correct folder, in this case using the Configuration object
    String listingFolder = config.getProperty("myApplication.listingFolder");
    // and use this parameter instead of the hard coded path
    File userList = new File(listingFolder, "users.txt"); // Compliant
    Collection<User> users = parse(userList);
    return users;
  }
}
```