<p>This rule raises an issue when a class implements the interface <code>java.lang.Cloneable</code>, but does not override the
<code>Object.clone()</code> method.</p>
<h2>Why is this an issue?</h2>
<p><code>Cloneable</code> is a <em>marker interface</em> that defines the contract of the <code>Object.clone</code> method, which is to create a
consistent copy of the instance. The <code>clone</code> method is not defined by the interface though, but by class <code>Objects</code>.</p>
<p>The general problem with marker interfaces is that their definitions cannot be enforced by the compiler because they have no own API. When a class
implements <code>Cloneable</code> but does not override <code>Object.clone</code>, it is highly likely that it violates the contract for
<code>Cloneable</code>.</p>
<h2>How to fix it</h2>
<p>Consider the following example:</p>

```java
class Foo implements Cloneable { // Noncompliant, override `clone` method
  public int value;
}
```
<p>Override the <code>clone</code> method in class <code>Foo</code>. By convention, it must call <code>super.clone()</code>. At this point, we know
that:</p>
<ul>
  <li> By behavioral contract, <code>Object.clone</code> will not throw a <code>CloneNotSupportedException</code>, because <code>Foo</code> implements
  <code>Cloneable</code>. </li>
  <li> The returned object is an instance of class <code>Foo</code> </li>
</ul>
<p>We can narrow down the return type of <code>clone</code> to <code>Foo</code> and handle the <code>CloneNotSupportedException</code> inside the
function instead of throwing it:</p>

```java
class Foo implements Cloneable { // Compliant

  public int value;

  @Override
  public Foo clone() {
    try {
      return (Foo) super.clone();
    } catch (CloneNotSupportedException e) {
      throw new AssertionError();
    }
  }
}
```
<p>Be aware that <code>super.clone()</code> returns a one-by-one copy of the fields of the original instance. This means that in our example, the
<code>Foo.value</code> field is not required to be explicitly copied in the overridden function.</p>
<p>If you require another copy behavior for some or all of the fields, for example, deep copy or certain invariants that need to be true for a field,
these fields must be patched after <code>super.clone()</code>:</p>

```java
class Entity implements Cloneable {

  public int id; // unique per instance
  public List<Entity> children; // deep copy wanted

  @Override
  public Entity clone() {
    try {
      Entity copy = (Entity) super.clone();
      copy.id = System.identityHashCode(this);
      copy.children = children.stream().map(Entity::clone).toList();
      return copy;
    } catch (CloneNotSupportedException e) {
      throw new AssertionError();
    }
  }
}
```
<p>Be aware that the <code>Cloneable</code> / <code>Object.clone</code> approach has several drawbacks. You might, therefore, also consider resorting
to other solutions, such as a custom <code>copy</code> method or a copy constructor:</p>

```java
class Entity implements Cloneable {

  public int id; // unique per instance
  public List<Entity> children; // deep copy wanted

  Entity(Entity template) {
    id = System.identityHashCode(this);
    children = template.children.stream().map(Entity::new).toList();
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://docs.oracle.com/javase/8/docs/api/java/lang/Cloneable.html">Interface Cloneable - Java™ Platform, Standard Edition 8 API
  Specification</a> </li>
  <li> <a href="https://docs.oracle.com/javase/8/docs/api/java/lang/Object.html#clone--">Object.clone - Java™ Platform, Standard Edition 8 API
  Specification</a> </li>
</ul>