<h2>Why is this an issue?</h2>
<p>A reference to <code>null</code> should never be dereferenced/accessed. Doing so will cause a <code>NullPointerException</code> to be thrown. At
best, such an exception will cause abrupt program termination. At worst, it could expose debugging information that would be useful to an attacker, or
it could allow an attacker to bypass security measures.</p>
<p>Note that when they are present, this rule takes advantage of nullability annotations, like <code>@CheckForNull</code> or <code>@Nonnull</code>,
defined in <a href="https://jcp.org/en/jsr/detail?id=305">JSR-305</a> to understand which values can be null or not. <code>@Nonnull</code> will be
ignored if used on the parameter of the <code>equals</code> method, which by contract should always work with null.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>The variable <code>myObject</code> is equal to <code>null</code>, meaning it has no value:</p>

```java
public void method() {
  Object myObject = null;
  System.out.println(myObject.toString()); // Noncompliant: myObject is null
}
```
<p>The parameter <code>input</code> might be <code>null</code> as suggested by the <code>if</code> condition:</p>

```java
public void method(Object input)
{
  if (input == null)
  {
    // ...
  }
  System.out.println(input.toString()); // Noncompliant
}
```
<p>The unboxing triggered in the return statement will throw a <code>NullPointerException</code>:</p>

```java
public boolean method() {
  Boolean boxed = null;
  return boxed; // Noncompliant
}
```
<p>Both <code>conn</code> and <code>stmt</code> might be <code>null</code> in case an exception was thrown in the try{} block:</p>

```java
Connection conn = null;
Statement stmt = null;
try {
  conn = DriverManager.getConnection(DB_URL,USER,PASS);
  stmt = conn.createStatement();
  // ...
} catch(Exception e) {
  e.printStackTrace();
} finally {
  stmt.close();  // Noncompliant
  conn.close();  // Noncompliant
}
```
<p>As <code>getName()</code> is annotated with <code>@CheckForNull</code>, there is a risk of <code>NullPointerException</code> here:</p>

```java
@CheckForNull
String getName() {...}

public boolean isNameEmpty() {
  return getName().length() == 0; // Noncompliant
}
```
<p>As <code>merge(…​)</code> parameter is annotated with <code>@Nonnull</code>, passing an identified potential null value (thanks to @CheckForNull)
is not safe:</p>

```java
private void merge(@Nonnull Color firstColor, @Nonnull Color secondColor) {...}

public void append(@CheckForNull Color color) {
  merge(currentColor, color);  // Noncompliant: color should be null-checked because merge(...) doesn't accept nullable parameters
}
```
<h4>Compliant solution</h4>
<p>Ensuring the variable <code>myObject</code> has a value resolves the issue:</p>

```java
public void method() {
  Object myObject = new Object();
  System.out.println(myObject.toString()); // Compliant: myObject is not null
}
```
<p>Preventing the non-compliant code to be executed by returning early:</p>

```java
public void method(Object input)
{
  if (input == null)
  {
    return;
  }
  System.out.println(input.toString()); // Compliant: if 'input' is null, this is unreachable
}
```
<p>Ensuring that no unboxing of <code>null</code> value can happen resolves the issue</p>

```java
public boolean method() {
  Boolean boxed = true;
  return boxed; // Compliant
}
```
<p>Ensuring that both <code>conn</code> and <code>stmt</code> are not <code>null</code> resolves the issue:</p>

```java
Connection conn = null;
Statement stmt = null;
try {
  conn = DriverManager.getConnection(DB_URL,USER,PASS);
  stmt = conn.createStatement();
  // ...
} catch(Exception e) {
  e.printStackTrace();
} finally {
  if (stmt != null) {
    stmt.close();  // Compliant
  }
  if (conn != null) {
    conn.close();  // Compliant
  }
}
```
<p>Checking the returned value of <code>getName()</code> resolves the issue:</p>

```java
@CheckForNull
String getName() {...}

public boolean isNameEmpty() {
  String name = getName();
  if (name != null) {
    return name.length() == 0; // Compliant
  } else {
    // ...
  }
}
```
<p>Ensuring that the provided <code>color</code> is not <code>null</code> resolves the issue:</p>

```java
private void merge(@Nonnull Color firstColor, @Nonnull Color secondColor) {...}

public void append(@CheckForNull Color color) {
  if (color != null) {
    merge(currentColor, color);  // Compliant
  }
}
```
<h2>Resources</h2>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/476">CWE-476 - NULL Pointer Dereference</a> </li>
  <li> CERT, EXP34-C. - <a href="https://wiki.sei.cmu.edu/confluence/x/QdcxBQ">Do not dereference null pointers</a> </li>
  <li> CERT, EXP01-J. - <a href="https://wiki.sei.cmu.edu/confluence/x/aDdGBQ">Do not use a null in a case where an object is required</a> </li>
</ul>