<p>Within IAM, identity-based policies grant permissions to users, groups, or roles, and enable specific actions to be performed on designated
resources. When an identity policy inadvertently grants more privileges than intended, certain users or roles might be able to perform more actions
than expected. This can lead to potential security risks, as it enables malicious users to escalate their privileges from a lower level to a higher
level of access.</p>
<h2>Why is this an issue?</h2>
<p>AWS Identity and Access Management (IAM) is the service that defines access to AWS resources. One of the core components of IAM is the policy
which, when attached to an identity or a resource, defines its permissions. Policies granting permission to an identity (a user, a group or a role)
are called identity-based policies. They add the ability to an identity to perform a predefined set of actions on a list of resources.</p>
<p>For such policies, it is easy to define very broad permissions (by using wildcard <code>"*"</code> permissions for example.) This is especially
true if it is not yet clear which permissions will be required for a specific workload or use case. However, it is important to limit the amount of
permissions that are granted and the amount of resources to which these permissions are granted. Doing so ensures that there are no users or roles
that have more permissions than they need.</p>
<p>If this is not done, it can potentially carry security risks in the case that an attacker gets access to one of these identities.</p>
<h3>What is the potential impact?</h3>
<p>AWS IAM policies that contain overly broad permissions can lead to privilege escalation by granting users more access than necessary. They may be
able to perform actions beyond their intended scope.</p>
<h4>Privilege escalation</h4>
<p>When IAM policies are too permissive, they grant users more privileges than necessary, allowing them to perform actions that they should not be
able to. This can be exploited by attackers to gain unauthorized access to sensitive resources and perform malicious activities.</p>
<p>For example, if an IAM policy grants a user unrestricted access to all S3 buckets in an AWS account, the user can potentially read, write, and
delete any object within those buckets. If an attacker gains access to this user’s credentials, they can exploit this overly permissive policy to
exfiltrate sensitive data, modify or delete critical files, or even launch further attacks within the AWS environment. This can have severe
consequences, such as data breaches, service disruptions, or unauthorized access to other resources within the AWS account.</p>
<h2>How to fix it in AWS CDK</h2>
<h3>Code examples</h3>
<p>In this example, the IAM policy allows an attacker to update the code of any Lambda function. An attacker can achieve privilege escalation by
altering the code of a Lambda that executes with high privileges.</p>
<h4>Noncompliant code example</h4>

```js
import { aws_iam as iam } from 'aws-cdk-lib'

new iam.PolicyDocument({
    statements: [new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        actions: ["lambda:UpdateFunctionCode"],
        resources: ["*"], // Noncompliant
    })],
});
```
<h4>Compliant solution</h4>
<p>The policy is narrowed such that only updates to the code of certain Lambda functions (without high privileges) are allowed.</p>

```js
import { aws_iam as iam } from 'aws-cdk-lib'

new iam.PolicyDocument({
    statements: [new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        actions: ["lambda:UpdateFunctionCode"],
        resources: ["arn:aws:lambda:us-east-2:123456789012:function:my-function:1"],
    })],
});
```
<h3>How does this work?</h3>
<h4>Principle of least privilege</h4>
<p>When creating IAM policies, it is important to adhere to the principle of least privilege. This means that any user or role should only be granted
enough permissions to perform the tasks that they are supposed to, and <em>nothing else</em>.</p>
<p>To successfully implement this, it is easier to start from nothing and gradually build up all the needed permissions. When starting from a policy
with overly broad permissions which is made stricter at a later time, it can be harder to ensure that there are no gaps that might be forgotten about.
In this case, it might be useful to monitor the users or roles to verify which permissions are used.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> AWS Documentation - <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#grant-least-priv">Policies and permissions
  in IAM: Grant least privilege</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> Rhino Security Labs - <a href="https://rhinosecuritylabs.com/aws/aws-privilege-escalation-methods-mitigation/">AWS IAM Privilege Escalation -
  Methods and Mitigation</a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/284">CWE-269 - Improper Privilege Management</a> </li>
</ul>