<h2>Why is this an issue?</h2>
<p>To optimize the rendering of React list components, a unique identifier (UID) is required for each list item. This UID lets React identify the item
throughout its lifetime. To provide it, use the <code>key</code> attribute of the list item. When the <code>key</code> attribute is missing, React
will default to using the item’s index inside the list component. If the element ordering changes, it will cause keys to not match up between renders,
recreating the DOM. It can negatively impact performance and may cause issues with the component state.</p>

```js
function Blog(props) {
  return (
    <ul>
      {props.posts.map((post) =>
        <li> <!-- Noncompliant: When 'posts' are reordered, React will need to recreate the list DOM -->
          {post.title}
        </li>
      )}
    </ul>
  );
}
```
<p>To fix it, use a string or a number that uniquely identifies the list item. The key must be unique among its siblings, not globally.</p>
<p>If the data comes from a database, database IDs are already unique and are the best option. Otherwise, use a counter or a UUID generator.</p>
<p>Avoid using array indexes since, even if they are unique, the order of the elements may change.</p>

```js
function Blog(props) {
  return (
    <ul>
      {props.posts.map((post) =>
        <li key={post.id}> <!-- Compliant: id will always be the same even if 'posts' order changes -->
          {post.title}
        </li>
      )}
    </ul>
  );
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a href="https://react.dev/learn/rendering-lists#rules-of-keys">Rendering lists</a> </li>
  <li> React Documentation - <a href="https://reactjs.org/docs/reconciliation.html#recursing-on-children">Recursing On Children</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/API/Crypto/randomUUID">Crypto: randomUUID() method</a> </li>
  <li> Wikipedia - <a href="https://en.wikipedia.org/wiki/Universally_unique_identifier">UUID</a> </li>
</ul>
<h3>Related rules</h3>
<ul>
  <li> <a href='checks/checks/?languages=JAVASCRIPT&tools=SONAR_LINT&rule=javascript:S6479'>javascript:S6479</a> - JSX list components should not use array indexes as key </li>
  <li> <a href='checks/checks/?languages=JAVASCRIPT&tools=SONAR_LINT&rule=javascript:S6486'>javascript:S6486</a> - JSX list components keys should match up between renders </li>
</ul>