<h2>Why is this an issue?</h2>
<p>React works in two phases: render and commit.</p>
<ul>
  <li> The render phase determines what changes need to be made to e.g. the DOM. During this phase, React calls render and then compares the result to
  the previous render. </li>
  <li> The commit phase is when React applies any changes. (In the case of React DOM, this is when React inserts, updates, and removes DOM nodes.)
  React also calls lifecycles like <code>componentDidMount</code> and <code>componentDidUpdate</code> during this phase. </li>
</ul>
<p>Render phase lifecycles include among others, the following lifecycle methods:</p>
<ul>
  <li> <code>componentWillMount</code> (or its alias <code>UNSAFE_componentWillMount</code>) </li>
  <li> <code>componentWillReceiveProps</code> (or its alias <code>UNSAFE_componentWillReceiveProps</code>) </li>
  <li> <code>componentWillUpdate</code> (or its alias <code>UNSAFE_componentWillUpdate</code>) </li>
</ul>
<p>These are considered unsafe and also happen to be the lifecycles that cause the most confusion within the React community and tend to encourage
unsafe coding practices.</p>

```js
class Foo extends React.Component {
  UNSAFE_componentWillMount() {}         // Noncompliant
  UNSAFE_componentWillReceiveProps() {}  // Noncompliant
  UNSAFE_componentWillUpdate() {}        // Noncompliant
}
```
<h2>How to fix it</h2>
<p>Instead of <code>componentWillUpdate</code>, use <code>getSnapshotBeforeUpdate</code> together with <code>componentDidUpdate</code>. The
<code>getSnapshotBeforeUpdate</code> lifecycle is called right before mutations are made. The return value for this lifecycle will be passed as the
third parameter to <code>componentDidUpdate</code>.</p>
<p>Instead of <code>componentWillReceiveProps</code>, Use <code>getDerivedStateFromProps</code> together with <code>componentDidUpdate</code>. The
<code>getDerivedStateFromProps</code> lifecycle is invoked after a component is instantiated as well as before it is re-rendered. It can return an
object to update state, or null to indicate that the new props do not require any state updates.</p>
<p>As for <code>componentWillMount</code>, React will call it immediately after the constructor. It only exists for historical reasons and should not
be used. Instead, use one of the alternatives:</p>
<ul>
  <li> To initialize state, declare <code>state</code> as a class field or set <code>this.state</code> inside the <code>constructor</code>. </li>
  <li> If you need to run a side effect or set up a subscription, move that logic to <code>componentDidMount</code> instead. </li>
</ul>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```js
class myComponent extends React.Component {
  constructor(props) {
      super(props);
  }

  componentWillMount() { // Noncompliant: "componentWillMount" is deprecated
      if (localStorage.getItem("token")) {
          this.setState({logged_in: true});
      }
  }
  // ...
}
```
<h4>Compliant solution</h4>

```js
class myComponent extends React.Component {
  constructor(props) {
      super(props);

      if (localStorage.getItem("token")) {
          this.setState({logged_in: true});
      }
  }
  // ...
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a
  href="https://react.dev/reference/react/Component#unsafe_componentwillmount"><code>UNSAFE_componentWillMount</code></a> </li>
  <li> React Documentation - <a
  href="https://react.dev/reference/react/Component#unsafe_componentwillreceiveprops"><code>UNSAFE_componentWillReceiveProps</code></a> </li>
  <li> React Documentation - <a
  href="https://react.dev/reference/react/Component#unsafe_componentwillupdate"><code>UNSAFE_componentWillUpdate</code></a> </li>
  <li> React Documentation - <a
  href="https://legacy.reactjs.org/blog/2018/03/27/update-on-async-rendering.html#migrating-from-legacy-lifecycles">Migrating from Legacy
  Lifecycles</a> </li>
</ul>