<h2>Why is this an issue?</h2>
<p>In JavaScript, a mutable variable is one whose value can be changed after it has been initially set. This is in contrast to immutable variables,
whose values cannot be changed once they are set.</p>
<p>Exporting mutable variables can lead to unpredictable behavior and bugs in your code. This is because any module that imports the variable can
change its value. If multiple modules import and change the value of the same variable, it can become difficult to track what the current value of the
variable is and which module changed it last.</p>
<h2>How to fix it</h2>
<p>If the value of the variable does not need to change, you can declare it as a constant using the <code>const</code> keyword. Alternatively, if you
have a group of related variables that need to be mutable, consider using a class to encapsulate them. You can then export an instance of the class,
or a factory function that creates instances of the class.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```js
let mutableVar = "initial value";

export { mutableVar }; // Noncompliant
```
<h4>Compliant solution</h4>

```js
const immutableVar = "constant value";
export { immutableVar };
```
<p>or</p>

```js
class MyClass {
  constructor() {
    this.mutableVar = "initial value";
  }
}

export function createMyClass() {
  return new MyClass();
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/let">let</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/const">const</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Glossary/Mutable">Mutable</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Glossary/Immutable">Immutable</a> </li>
</ul>