<h2>Why is this an issue?</h2>
<p>When you call a function in JavaScript and provide more arguments than the function expects, the extra arguments are simply ignored by the
function.</p>

```js
function sum(a, b) {
  return a + b;
}

sum(1, 2, 3); // Noncompliant: The last argument is unexpected and will be ignored
```
<p>Passing extra arguments in JavaScript is not inherently "bad," but it can lead to some potential issues or confusion if not handled correctly:</p>
<ul>
  <li> The function signature is an essential part of its interface. Passing extra arguments can obscure the function’s intended use and make it less
  clear what the function actually requires. </li>
  <li> This can lead to unexpected behavior, as the function might not work as intended or produce incorrect results. </li>
  <li> Code that passes extra arguments can become harder to understand and maintain, especially when revisiting it at a later time. </li>
  <li> Other developers might find it challenging to comprehend the function’s purpose if extra arguments are scattered throughout the codebase. </li>
  <li> If you refactor the function later or rely on an external library that changes the expected number of arguments, your code with extra arguments
  could break unexpectedly. </li>
</ul>
<p>While it’s possible to pass extra arguments, it’s essential to note that accessing those extra arguments directly inside the function is not
straightforward. One common approach to handling extra arguments is to use the <code>arguments</code> object, which is an array-like object available
within all function scopes.</p>

```js
function sum() {
  let total = 0;
  for (let i = 0; i < arguments.length; i++) {
    total += arguments[i];
  }
  return total;
}

sum(1, 2, 3); // Compliant
```
<p>However, it’s generally recommended to use the rest parameter syntax (<code>...args</code>) or utilize other techniques like the spread operator to
deal with variable numbers of arguments in a more readable and maintainable way.</p>

```js
function sum(...args) {
  return args.reduce((a,b) => a + b, 0);
}

sum(1, 2, 3); // Compliant
```
<h3>Exceptions</h3>
<p>No issue is reported when <code>arguments</code> is used in the body of the function being called.</p>

```js
function doSomething(a, b) {
  compute(arguments);
}

doSomething(1, 2, 3); // Compliant
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Functions/arguments">The <code>arguments</code>
  object</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Functions/rest_parameters">Rest parameters</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Spread_syntax">Spread syntax
  (<code>...</code>)</a> </li>
</ul>