<h2>Why is this an issue?</h2>
<p>Ignoring function parameters or overwriting them with a new value without reading them can lead to confusion and errors in the code. Developers
won’t be able to tell whether the original parameter or some temporary variable is being accessed without going through the whole function. It may
indicate that the function is not properly designed or that there is a mistake in the code.</p>
<p>Moreover, some developers might also expect assignments of function parameters to be visible to callers, which is not the case. Arguments are
always passed by value and never passed by reference. If a function reassigns a parameter, the value won’t change outside the function. It is not
possible to simulate an assignment on that variable in the caller’s scope. However, objects are passed by value to their reference (<a
href="https://en.wikipedia.org/wiki/Evaluation_strategy#Call_by_sharing">passed by sharing</a>), which means if the object’s properties are mutated,
the change will impact the outside of the function.</p>
<p>The same logic applies to caught exceptions and variable declarations inside <code>for...in</code> and <code>for...of</code> statements: their
initial values should not be ignored.</p>

```ts
function myFunction(name, strings) {
  name = foo; // Noncompliant: initial value of 'name' is ignored

  for (let str of strings) {
    str = "";  // Noncompliant: initial value of 'str' is ignored
  }
}
```
<p>Function parameters, caught exceptions, and variables initialized in <code>for...in</code> and <code>for...of</code>statements should be read at
least once before reassigning them. If they do not need to be read, the code should be refactored to avoid confusion.</p>

```ts
function myFunction(name, strings) {
  const nameCopy = name;
  name = foo;

  for (let str of strings) {
    const strCopy = str;
    str = "";
  }
}
```
<h3>Exceptions</h3>
<p>There is a common pattern in JavaScript to overwrite certain parameters depending on other parameters that are optional. For example, a callback
is, by convention, always passed in the last position. If a parameter in a previous position was not passed, the callback will be passed in its
position instead.</p>
<p>Therefore, the rule ignores parameter reassignments that are inside an <code>if</code> statement block.</p>

```ts
function myFunction(param, optionalParam, cb) {
  if (typeof optionalParam === 'function') {
    cb = optionalParam;
    optionalParam = {};
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Functions">Functions</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/try...catch#catch_binding">Catch
  binding</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/for...of"><code>for...of</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/for...in"><code>for...in</code></a> </li>
  <li> Wikipedia - <a href="https://en.wikipedia.org/wiki/Evaluation_strategy#Call_by_sharing">Call by sharing</a> </li>
</ul>